function recess = window(y, window,  tol)
%this function classifies recessions by finding the peaks and troughs
%of a series through the use of a rolling window.  If the peak/trough is
%not unique, recessions are defined as the first falling observation to the
%last observation before the series turns up again.

%arguments: y  - series to analyze
%           window - width of window
%           tol  - tolerance for a "soft" max/min.  A peak, for example, is
%           identified if x(t) >= max{x(t-window),....,x(t+window)} - tol.
%           Default for tol is 0.

if nargin < 3
    tol = 0;
end

debug = false;

T = length(y);

recess = zeros(T,1);


%find the peaks and troughs with the standard window size
[peak trough] = windowScan(y,window,tol);

%We now clean the turning points to make sure they alternate between peaks
%and troughs.

%If there is a point t that is a peak and a trough then the function must
%be constant on [t-w, t+w].  t+1 is therefore also either a peak or a
%trough because either y(t+w+1)>y(t) and it is a trough or vice versa or
%the function is still constant and t+1 is both a peak and a trough.  In
%light of this logic, any point that is both a peak and trough is recoded
%to be neither.
peak = peak - (peak .* trough);
trough = trough - (peak .* trough);

if debug
    load data/asym1;
    disp([[1:length(peak)]' time peak trough]);
end

%Ensure alternating peaks and troughs by eliminating the first of
%consecutive peaks (troughs).
current = 0;
lastIndex = [];
for t = 1:T
    if peak(t)==1 && current == 1
        peak(lastIndex) = 0;
    elseif trough(t) == 1 && current == -1
        trough(lastIndex) = 0;
    end
    
    if peak(t) == 1
        current = 1;
        lastIndex = t;
    elseif trough(t) == 1;
        current = -1;
        lastIndex = t;
    end
end



recess = cumsum(peak - trough);
if sum(recess) < 0
    recess = recess + 1;
end




%------------------------------------

function [peak trough] = windowScan(y,window,tol)

    T = length(y);

    trough = zeros(T,1);
    peak = zeros(T,1);
    for i=(window+1):(T-window)         %avoid ends of sample
        H = min(T,i+window);
        L = max(1,i-window);
        trough(i) = y(i) <= (min(y(i:H))+tol) & y(i) <= (min(y(L:i))+tol);
        peak(i) = y(i) >= (max(y(i:H))-tol) & y(i) >= (max(y(L:i))-tol);
    end
%end of function

