%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% TransfersMainLucas.m
% Main script file to compute the steady state and the transition dynamics 
% of Section 4 in Oh and Reis (2011), 
% now with fixed capital and positive final goods firm profits. 
% Savings decisions are now by trading the claims for these profits.
% 
% Last modified: May 17, 2011                                              
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%----------------------------------------------------
% Period individual utility function: 
% U(c,h) = log(c) -chi*(1-h)^(phi)*n   
% [ h=1 with prob ppi and h~U(0,eta) with prob (1-ppi), n = {0,1} ]
% Period aggregate production function: 
% F(K,X) = A * K^(alpha) * X^(1-alpha)
%
% Individual Productivity Process
% log(sp) = zmean*(1-rho) + rho*log(s) + stdshock*ep
% ep ~ N(0,1)
%----------------------------------------------------

%%%%%%%%%%%%%%%% Step 1: Parameter Specification %%%%%%%%%%%%%%%%
%--------------------------- User Setup ------------------------%
beta  = 0.85;      % annual discount factor (benchmark: 0.96)
uform = 'CRRA';    % utility function on consumption (benchmark: 'CRRA')
gamma = 1;         % risk aversion coefficient (benchmark : 1)
alpha = 0.36;      % capital share (benchmark: 0.36)
chi = 2.2001;      % disutility of labor supply (benchmark: 2.2001)
eta = 0.32;        % maximum healthiness for Neoclassicals (benchmark: 0.32) 
phi = 1;           % curvature (benchmark: 1)
delta = 0;         % annual depreciation ratio (benchmark: 0 when K is fixed)
mu = 1.25;         % average markup in the US economy (benchmark: 1.25)
ppi = 0.51;        % probability of h = 1 (benchmark = 0.51)
rho = 0.9;         % annual persistence of idiosyncratic salary offer (benchmark: 0.9)
stdshock = 0.25;   % annual std of the idiosyncratic shock (benchmark: 0.25)
zmean = -stdshock^2/(2*(1-rho^2)); % mean of log(s) for E(s) = 1 to hold
G = 0;             % G/GDP (benchmark: 0)
A = 1;             % aggregate productivity (benchmark: 1)
K = 3.4227;        % aggregate capital stock (benchmark: 2.7908 in variable capital)
%---------------------------------------------------------------%

%%%%%%%%%%%%%%%% Step 2: Setting State Space Grid %%%%%%%%%%%%%%%%
%--------------------------- User Setup ------------------------%
nk = 191;          % number of gridpoints for capital (benchmark: 191)
klow = 0;          % minimum value of the capital grid
khigh = 65;        % maximum value of the capital grid
ns = 21;           % number of gridpoints for idiosyncratic shock process
nh = 21;           % number of gridpoints for mixed uniform iid health shock process 
kmethod = 2;       % choose how you want to space the capital grid:
                   % 1 = Equally spaced
                   % 2 = Logarithmic spacing (recommended)
                   % 3 = Chebyshev nodes
smethod = 2;       % choose how you want to space the AR(1) idiosyncratic grid:
                   % 1 = Tauchen Hussey with Floden modification for
                   %     persistent shock
                   % 2 = Tauchen Hussey
                   % 3 = Tauchen (caution: zmean = 0 imposed in this case)
%---------------------------------------------------------------%

% Build Grid for Capital 
if kmethod == 1;                     % grid when equally spaced
    kgrid = linspace(klow,khigh,nk)';
elseif kmethod == 2;                 % grid when logarithmically spaced
    kgridtemp = linspace(0,log(khigh+1-klow),nk)';
    kgrid = exp(kgridtemp)-1+klow;
elseif kmethod == 3;                 % Chebyshec collocation nodes.
    ZZ = -cos((2*[1:nk]'-1)*pi/(2*nk)); 
    kgrid = (ZZ+1)*(khigh-klow)/2+klow;
end

% Build Grid for Idiosyncratic Shock Process
if smethod == 1        % Tauchen Hussey with Floden modification
    % baseSigma: refinement suggested by Martin Floden for his code tauchenhussey.m
    baseSigma = (0.5+rho/4)*stdshock+(0.5-rho/4)*(stdshock/sqrt(1-rho^2)); 
    [logsgrid,sprob] = tauchenhussey(ns,zmean,rho,stdshock,baseSigma);
elseif smethod == 2    % original Tauchen Hussey
    [logsgrid,sprob] = tauchenhussey(ns,zmean,rho,stdshock,stdshock);
elseif smethod == 3    % Tauchen (1986)
    [sprob,logsgrid] = markovappr(rho,stdshock,3,ns);
end
sgrid = exp(logsgrid); % convert back to the original variable (not the log process!)
sgrid = sgrid(:); % column vector

% Stationary distribution of the markov idiosyncratic process
sp = limitdist1(sprob); 
% CDF of s
cdfs = cumsum(sp);

% Build Grid for iid Uniform Health Shock Process
hgrid = [linspace(0,eta,nh-1) 1]';
if ppi == 0 % neoclassical
    hp = (1/nh)*ones(1,nh);
else % Keynesian (of course ppi should be much greater than (1/nh) for discretization to make sense)
    hp = ((1-ppi)/(nh-1))*ones(1,nh); % partially healthy w/ uniform  
    hp(nh) = ppi; % totally healthy w/ prob: ppi
end

%%%%%%%%%%%%%%%%% Step 3: Transfers Policy Setup %%%%%%%%%%%%%%%%%
% Transfers policy (endogenous calibration of coefficients)
[ Sm Hm ] = ndgrid(sgrid,hgrid);
%--------------------------- User Setup ------------------------%
xx=0.45/0.55;    % ratio of health transfers to income transfers expenditure
scut = 0.05;     % proportion of people not receiving transfers for income
sval = exp(interp1(cdfs,log(sgrid),1-scut)); % threshold of income not receiving transfers (set to be same as health)
Spolicy = ((1/sval)*(sval-Sm)).*(Sm<=sval);  % income transfers expenditure function
Hpolicy = ((3/(4*eta))*((4/3)*eta-Hm)).*(Hm<=eta); % health transfers expenditure function
tyratio = 0.126; % transfers to output ratio
%---------------------------------------------------------------%
% If no transfers
%--------------------------- User Setup ------------------------%
Spolicy = 0*Sm; Hpolicy=0*Hm;        % counterfactual no transfers policy
%---------------------------------------------------------------%

%%%%%%%% Step 4: Store the Specification into a Structural Form %%%%%%%% 
PARA.beta = beta; PARA.uform = uform; PARA.gamma = gamma;
PARA.alpha = alpha; PARA.chi = chi; PARA.phi = phi; PARA.delta = delta; 
PARA.mu = mu; PARA.G = G; PARA.A = A; PARA.K = K; 
PARA.nk = nk; PARA.ns = ns; PARA.nh = nh; 
PARA.kgrid = kgrid; PARA.sgrid=sgrid; PARA.hgrid=hgrid; 
PARA.sprob=sprob; PARA.sp = sp; PARA.hp=hp;
PARA.xx=xx; PARA.Spolicy=Spolicy; PARA.Hpolicy=Hpolicy; PARA.tyratio=tyratio;

%%%%%% Step 5: Searching over (X) that is Consistent %%%%%% 
%--------------------------- User Setup ------------------------%
tol1 = 1e-7;          % tolerance for market clearing (X,r)
X0 = 0.467773304687;
r0 = 0.036843008656;
%X0 = 0.46432;        % interpolation
%r0 = 0.037126;       % interpolation
psi0 = 0.20;          % smoothing parameter for convergence 
%---------------------------------------------------------------%
Xnew = X0; rnew = r0;     % initial value setup


% 1: approximate solution search by using discretization method
disp('Approximation by Discretization Method')
iter = 1;
[fval VAR] = VFksh01Lucas(Xnew,rnew,PARA)
while (fval > tol1) && (iter <=30)
    Xnew = Xnew + psi0*(VAR.Xnew-Xnew);
    rnew = rnew + psi0*(VAR.rnew-rnew);
    [fval VAR] = VFksh01Lucas(Xnew,rnew,PARA);
    disp('iter   fval   X    r')
    disp([' ' num2str(iter) '  ' num2str(fval) '  ' num2str(Xnew) '  ' num2str(rnew)])
    iter = iter + 1;
end

%{
% 2: precise solution search by using piecewise linear method
% to utilize parfor: "matlabpool open 4"
disp('Precise Search by Piecewise Linear Method')
[fval VAR] = VFksh02LucasParfor(Xnew,rnew,PARA);
iter = 1;  
while (fval > tol1) && (iter <=30)
    Xnew = Xnew + psi0*(VAR.Xnew-Xnew);
    rnew = rnew + psi0*(VAR.rnew-rnew);
    [fval VAR] = VFksh02LucasParfor(Xnew,rnew,PARA);
    disp('iter   fval   X    r')
    disp([' ' num2str(iter) '  ' num2str(fval) '  ' num2str(Xnew) '  ' num2str(rnew) ])
    iter = iter + 1;
end
%}

disp('X   r')
disp([num2str(VAR.Xnew) '  ' num2str(VAR.rnew)])

%%%%%%%%% Step 6: Transition Dynamics Setup %%%%%%%%% 
%------------------ User Setup -------------------%
VAR0=VAR;          % 1: Initial State
VAR1=VAR0;         % 2: Future Steady State
nt = 220;          % maximum time path
psi0 = 0.20;       % smoothing parameter for updating
tol = 1e-4;        % tolerance level for time path convergence
nq = 50;           % maximum iteration for time path convergence
tol1 = 1e-2;       % tolerance level for golden section search
%-------------------------------------------------%

%%%%%%%%%%%%%%% Step 7: Policy Setup %%%%%%%%%%%%%%%
% 1: Government shock
Gt = G*ones(nt,1);        % time path for govt spending
%{
%------------------ User Setup -------------------%
rhoG = 0.9;                     % persistence of G
Gshock = (1-rhoG)*0.010*VAR0.Y; % shock to G
for it = 2:nt-1
    Gt(it) = rhoG^(it-2)*Gshock;
end
%-------------------------------------------------%
%}

% 2: Transfers Policy (Endogenous Calibration of Coefficients)
%------------------ User Setup -------------------%
trmethod = 4;  % choose the correct transfers policy specification
               % 0 = No shock for transfers
               % 1 = Systematic Transfers Policy Shock (figure 8)
               % 2 = Neoclassical Transfers Shock (figure 5)
               % 3 = Keynesian Transfers Shock (figure 6)
               % 4 = Nontargeted Transfers Shock (figure 7)               
%-------------------------------------------------%
Tpolicyt=zeros(ns,nh,nt); Tsumt=zeros(nt,1); 
taut=zeros(nt,1); Tt=zeros(ns,nh,nt);
gam_s = VAR0.gam_s; gam_h = VAR0.gam_h; 
gam_st=gam_s*ones(nt,1); gam_ht=gam_h*ones(nt,1);
for it=1:nt
    Tpolicyt(:,:,it) = gam_st(it)*Spolicy + gam_ht(it)*Hpolicy; % steady state transfers policy
end

switch trmethod
    case 0 % no shock
        % do nothing!!
        
    case 1 % 2.1: Systematic Transfers Policy Shock
    %------------------ User Setup -------------------%
    rhoT = 0;     % persistence of transfers coefficient
    tr_y = 0.034; % total amount of shock intended in terms of output
    tr_h = 0.3;   % ratio of transfers increased by health
    %-------------------------------------------------%
    gam_sshock = gam_s*(1+(1-rhoT)*(1-tr_h)*(1+xx)*(tr_y/tyratio));
    gam_hshock = gam_h*(1+(1-rhoT)*tr_h*(1+xx)*(tr_y/tyratio));
    gam_st(2)=gam_sshock; gam_ht(2)=gam_hshock;
    for it=3:nt-1
        gam_st(it)=gam_s*(gam_st(it-1)/gam_s)^rhoT;  
        gam_ht(it)=gam_h*(gam_ht(it-1)/gam_h)^rhoT;
    end
    for it=1:nt
        Tpolicyt(:,:,it) = gam_st(it)*Spolicy + gam_ht(it)*Hpolicy; % transfers policy
    end

    case 2 % 2.2: Neoclassical Transfers Policy Shock
    %------------------ User Setup -------------------%
    tr_y = 0.034;     % total amount of shock intended in terms of output
    hrich = (11:20);  % healthy people who give up transfers 
    hpoor = (1:10);   % unhealthy people who receive transfers
    srich = (13:21);  % efficient people who give up transfers
    spoor = (1:9);    % inefficient people who receive transfers
    %-------------------------------------------------%
    coverage1 = sum(hp(hrich))*sum(sp(srich)); % total distribution of rich people
    coverage2 = sum(hp(hpoor))*sum(sp(spoor)); % total distribution of poor people
    trratio = tr_y/coverage1; % amount of transfers taken from the rich 
    trval = trratio*VAR0.Y;   % amount of shock
    multi = coverage1/coverage2; % multiplier to balance the budget
    % rich to poor transfers
    %------------------ User Setup -------------------%
    Tpolicyt(srich,hrich,2) = Tpolicyt(srich,hrich,2) - trval;
    Tpolicyt(spoor,hpoor,2) = Tpolicyt(spoor,hpoor,2) + multi*trval;
    %-------------------------------------------------%

    case 3 % 2.3: Keynesian Transfers Policy Shock
    %------------------ User Setup -------------------%
    tr_y = 0.034;     % total amount of shock intended in terms of output
    hrich = nh;       % healthy people who give up transfers
    hpoor = nh;       % unhealthy people who receive transfers
    srich = (12:21);  % efficient people who give up transfers
    spoor = (1:10);   % inefficient people who receive transfers
    %-------------------------------------------------%
    coverage1 = sum(hp(hrich))*sum(sp(srich)); % total distribution of rich people
    coverage2 = sum(hp(hpoor))*sum(sp(spoor)); % total distribution of poor people
    trratio = tr_y/coverage1; % amount of transfers taken from the rich 
    trval = trratio*VAR0.Y;   % amount of shock
    multi = coverage1/coverage2; % multiplier to balance the budget
    % rich to poor transfers
    %------------------ User Setup -------------------%
    Tpolicyt(srich,hrich,2) = Tpolicyt(srich,hrich,2) - trval;
    Tpolicyt(spoor,hpoor,2) = Tpolicyt(spoor,hpoor,2) + multi*trval;
    %-------------------------------------------------%

    case 4 % 2.4: Nontargeted Transfers Policy Shock
    %------------------ User Setup -------------------%
    tr_y = 0.034;     % total amount of shock intended in terms of output
    hrich = (nh);     % healthy people who give up transfers 
    hpoor = (1:10);   % unhealthy people who receive transfers
    srich = (13:21);  % efficient people who give up transfers
    spoor = (1:11);   % inefficient people who receive transfers
    %-------------------------------------------------%
    coverage1 = sum(hp(hrich))*sum(sp(srich)); % total distribution of rich people
    coverage2 = sum(hp(hpoor))*sum(sp(spoor)); % total distribution of poor people
    trratio = tr_y/coverage1; % amount of transfers taken from the rich 
    trval = trratio*VAR0.Y;   % amount of shock
    multi = coverage1/coverage2; % multiplier to balance the budget
    % rich to poor transfers
    %------------------ User Setup -------------------%
    Tpolicyt(srich,hrich,2) = Tpolicyt(srich,hrich,2) - trval;
    Tpolicyt(spoor,hpoor,2) = Tpolicyt(spoor,hpoor,2) + multi*trval;
    %-------------------------------------------------%
end

for it=1:nt
    Tsumt(it) = ( Tpolicyt(:,:,it)*hp' )'*sp';
    taut(it) = Tsumt(it) + Gt(it); % total lump-sum tax to balance the budget
    Tt(:,:,it) = Tpolicyt(:,:,it)-taut(it); % net transfers (transfers - lumpsum taxes)
end

% 3: TFP shock
At = A*ones(nt,1);    % time path for technology shock
%{
%------------------ User Setup -------------------%
rhoA = 0.66;              % persistence of log(A)
Ashock = 0.01;            % shock to log(A/Ass)
At(2) = A*exp(Ashock);
for it = 3:nt-1
    At(it) = A*exp(rhoA*log(At(it-1)/A));
end
%-------------------------------------------------%
%}

% 4. Sticky Information
LAMBt = ones(nt,1);
%------------------ User Setup -------------------%
LAMBshock = 0.5;          % stickiness parameter
for it = 2:nt-1
    LAMBt(it) = 1-(1-LAMBshock)^(it-1);
end
%-------------------------------------------------%

% 5. Monetary Policy
zt = ones(nt,1);
%------------------ User Setup -------------------%
mpolicy = 1; % monetary policy setup
             % 1 = strict price targeting 
             % 2 = nominal income targeting
             % 3 = forward-looking Taylor rule
TRco = 2;    % Taylor Rule coefficient if Taylor rule is chosen
%-------------------------------------------------%

%%%%%%%%%%%%% Step 8: Steady State Setup %%%%%%%%%%%%%
% Initial State (VAR0)
Z_old = VAR0.VF;           % value function (average over h)
EZ_old = VAR0.VF2;         % expected value function
gk_old = VAR0.gk;          % distribution (k,e)
ndec_old = VAR0.ndec;      % labor decision
kdec_old = VAR0.kdec;      % capital decision
cdec_old = VAR0.cdec;      % consumption decision
r_old = rnew;              % real interest rate
w_old = VAR0.wnew;         % wage
p_old = VAR0.pnew;         % MPX
d_old = VAR0.dnew;         % dividend
X_old = Xnew;              % aggregate intermediate good
KXratio_old = K/X_old;     % K/X ratio 
E_old = VAR0.Enew;         % aggregate labor
L_old = VAR0.Lnew;         % aggregate effective labor
T_old = VAR0.T;            % net transfers
pk_old = VAR0.pknew;       % price of real profit share
Rk_old = VAR0.Rknew;       % real profit (final goods firm)

% Future Steady State (VAR1)
Z_new = VAR1.VF;           % value function (average over h)
EZ_new = VAR1.VF2;         % expected value function
gk_new = VAR1.gk;          % distribution (k,e)
ndec_new = VAR1.ndec;      % labor decision
kdec_new = VAR1.kdec;      % capital decision
cdec_new = VAR1.cdec;      % consumption decision
r_new = rnew;              % real interest rate
w_new = VAR1.wnew;         % wage
p_new = VAR1.pnew;         % MPX
d_new = VAR1.dnew;         % dividend
X_new = Xnew;              % aggregate intermediate good
KXratio_new = K/X_new;     % K/X ratio 
E_new = VAR1.Enew;         % aggregate labor
L_new = VAR1.Lnew;         % aggregate effective labor
T_new = VAR1.T;            % net transfers
pk_new = VAR1.pknew;       % price of real profit share
Rk_new = VAR1.Rknew;       % real profit (final goods firm)

%%%%%%%%%%%%%% Step 9: Set up the time path %%%%%%%%%%%%%%
Zt  = zeros(nk,ns,nt);     % value function
EZt = zeros(nk,ns,nt);     % expected value function
gkt = zeros(nk,ns,nt);     % distribution (k,s)
ndect = zeros(nk,ns,nt);   % labor supply decision
kdect = zeros(nk,ns,nt);   % capital decision 
cdect = zeros(nk,ns,nt);   % consumption decision
rt = zeros(nt,1);          % real interest rate
wt = zeros(nt,1);          % wage
pt = zeros(nt,1);          % MPX
dt = zeros(nt,1);          % dividend
Xt = zeros(nt,1);          % aggregate intermediate good
KXratiot = zeros(nt,1);    % K/X ratio
Et = zeros(nt,1);          % aggregate labor
Lt = zeros(nt,1);          % aggregate effective labor
pkt = zeros(nt,1);         % price of real profit share
Rkt = zeros(nt,1);         % real profit

% Plug in initial state to the time path
Zt(:,:,1)=Z_old; EZt(:,:,1)=EZ_old; gkt(:,:,1)=gk_old/sum(sum(gk_old)); 
ndect(:,:,1)=ndec_old; kdect(:,:,1)=kdec_old; cdect(:,:,1)=cdec_old;
rt(1)=r_old; wt(1)=w_old; pt(1)=p_old; dt(1)=d_old; 
Xt(1)=X_old; Lt(1)=L_old; pkt(1)=pk_old; Rkt(1)=Rk_old;
KXratiot(1)=KXratio_old; Et(1)=E_old; 
% Plug in future steady state to the time path
Zt(:,:,nt)=Z_new; EZt(:,:,nt)=EZ_new; gkt(:,:,nt)=gk_new/sum(sum(gk_new)); 
ndect(:,:,nt)=ndec_new; kdect(:,:,nt)=kdec_new; cdect(:,:,nt)=cdec_new;
rt(nt)=r_new; wt(nt)=w_new; pt(nt)=p_new; dt(nt)=d_new; 
Xt(nt)=X_new; Lt(nt)=L_new; pkt(nt)=pk_new; Rkt(nt)=Rk_new;
KXratiot(nt)=KXratio_new; Et(nt)=E_new; 

%%%%%%%%%%%%%% Step 10: Guess a sequence for (X,r,pk) %%%%%%%%%%%%%%
rt = linspace(rt(1),rt(nt),nt)'; % real interest rate guess for time path
Xt(1:nt) = linspace(Xt(1),Xt(nt),nt)'; % intermediate good guess 
rtnew = rt; Xtnew = Xt; 
% need an initial value for pk as well for the recursive structure
pkt = linspace(pkt(1),pkt(nt),nt)';
pkt(nt-1)=pkt(nt);
% capital (just for consistency)
Kt = K*ones(nt,1);

%%%%% Step 11: Iteration to approximate time path for (X,r) %%%%%
clear Sm Hm
[ Km Sm Hm ] = ndgrid(kgrid,sgrid,hgrid);

disp('start running transition')
q=0; crit = 1;
while (crit>=tol) && (q < nq);
    tic; 
    % time path for (KXratio,w,d,p) given sequence (X,r)  
    KXratiot = K./Xt; % K/X ratio
    pt = (1-alpha)*At.*(KXratiot).^(alpha); % marginal cost of aggregate intermediate good
    
    %---------------- Monetary Policy ----------------%
    switch mpolicy
        case 1 % Strict Inflation Targeting
            zt = ones(nt,1);
        case 2 % Nominal Income Targeting
            zt = 1./(At.*(KXratiot.^(alpha)).*Xt);
        case 3 % Forward-looking Taylor Rule
            zt(1) = 1; 
            for iz = 2:nt-1; 
                zt(iz) = zt(iz-1)*((1+rt(iz-1)-delta)./(1+r_old-delta)).^(1/(TRco-1)); 
            end
    end
    %-------------------------------------------------%
    
    wt_up = (pt./mu).^(1/(1-mu)) - (1-LAMBt).*(w_old*zt(1)./zt).^(1/(1-mu));
    wt_up((wt_up<=0))=1e-5; % force small positive value if negative value is assigned (possible during the guessing process)
    wt = (wt_up./LAMBt).^(1-mu); % wage
    
    elt_up   = LAMBt.*(wt.*zt/(w_old*zt(1))).^(mu/(1-mu)) + 1 - LAMBt;
    elt_down = LAMBt.*(wt.*zt/(w_old*zt(1))).^(1/(1-mu))  + 1 - LAMBt;
    elt = Xt.*elt_up./((elt_down).^(1-mu));
    
    dt_up   = LAMBt.*(wt/w_old).^(mu/(1-mu))+(1-LAMBt).*(mu*w_old./wt-1)/(mu-1);
    dt_up((dt_up<0)) = 0;
    dt_down = LAMBt.*(wt/w_old).^(mu/(1-mu))+(1-LAMBt);
    dt = (mu-1)*wt.*elt.*dt_up./dt_down; % dividend
    
    % Backwards Iteration of the policy function/value function
    clear Vh decisn kdec1 kdec2 cdec1 cdec2
    Vh=zeros(nk,ns,nh); decisn=zeros(nk,ns,nh); 
    kdec1=zeros(nk,ns,nh); kdec2=zeros(nk,ns,nh);
    cdec1=zeros(nk,ns,nh); cdec2=zeros(nk,ns,nh);
    
    for it=nt-1:-1:2 % backward iteration !!!
        EZ_it=EZt(:,:,it+1); r_it=rt(it); w_it=wt(it); T_it=Tt(:,:,it); d_it=dt(it); 
        parfor ih=1:nh
            kdec1_ks=zeros(nk,ns); kdec2_ks=zeros(nk,ns);
            Vh_ks=zeros(nk,ns); decisn_ks=zeros(nk,ns);
            cdec1_ks=zeros(nk,ns); cdec2_ks=zeros(nk,ns);
            for is=1:ns
                for ik=1:nk
                    % Case 1: n=0 (no work)
                    np = 0;
                    cp = (1-delta+r_it)*kgrid(ik) + sgrid(is)*w_it*np + T_it(is,ih) + d_it - kgrid;
                    vfh = utilf(cp,uform,gamma) - np*chi*(1-hgrid(ih))^(phi) + beta*EZ_it(:,is);
                    [Vh1, decis1] = max ( vfh ) ;
                    kdec1_ks(ik,is) = kgrid(decis1);
                    cdec1_ks(ik,is) = (1-delta+r_it)*kgrid(ik) + sgrid(is)*w_it*np + T_it(is,ih) + d_it - kdec1_ks(ik,is);
                    %{
                    % local search of kp using golden section search 
                    ikp = decis1; % to make the notation convenient
                    if ikp == 1
                        kpmin=kgrid(1); kpmax=kgrid(2);
                    elseif ikp == nk
                        kpmin=kgrid(nk-1); kpmax=kgrid(nk);
                    else
                        kpmin=kgrid(ikp-1); kpmax=kgrid(ikp+1);
                    end
                    kdec1_ks(ik,is) = gss1(@(kp) nvaluefparfor(kp,np,ik,is,ih,delta,r_it,w_it,T_it,d_it,beta,chi,phi,uform,gamma,EZ_it,kgrid,sgrid,hgrid),kpmin,kpmax,1e-6);
                    Vh1 = -nvaluefparfor(kdec1_ks(ik,is),np,ik,is,ih,delta,r_it,w_it,T_it,d_it,beta,chi,phi,uform,gamma,EZ_it,kgrid,sgrid,hgrid);
                    cdec1_ks(ik,is) = (1-delta+r_it)*kgrid(ik) + sgrid(is)*w_it*np + T_it(is,ih) + d_it - kdec1_ks(ik,is);
                    %}
                    
                    % Case 2: n=1 (work)
                    np = 1;
                    cp = (1-delta+r_it)*kgrid(ik) + sgrid(is)*w_it*np + T_it(is,ih) + d_it - kgrid;
                    vfh = utilf(cp,uform,gamma) - np*chi*(1-hgrid(ih))^(phi) + beta*EZ_it(:,is);
                    [Vh2, decis2] = max ( vfh ) ;
                    kdec2_ks(ik,is) = kgrid(decis2);
                    cdec2_ks(ik,is) = (1-delta+r_it)*kgrid(ik) + sgrid(is)*w_it*np + T_it(is,ih) + d_it - kdec2_ks(ik,is);
                    %{
                    % local search of kp using golden section search 
                    ikp = decis2; % to make the notation convenient
                    if ikp == 1
                        kpmin=kgrid(1); kpmax=kgrid(2);
                    elseif ikp == nk
                        kpmin=kgrid(nk-1); kpmax=kgrid(nk);
                    else
                        kpmin=kgrid(ikp-1); kpmax=kgrid(ikp+1);
                    end
                    kdec2_ks(ik,is) = gss1(@(kp) nvaluefparfor(kp,np,ik,is,ih,delta,r_it,w_it,T_it,d_it,beta,chi,phi,uform,gamma,EZ_it,kgrid,sgrid,hgrid),kpmin,kpmax,1e-6);
                    Vh2 = -nvaluefparfor(kdec2_ks(ik,is),np,ik,is,ih,delta,r_it,w_it,T_it,d_it,beta,chi,phi,uform,gamma,EZ_it,kgrid,sgrid,hgrid);
                    cdec2_ks(ik,is) = (1-delta+r_it)*kgrid(ik) + sgrid(is)*w_it*np + T_it(is,ih) + d_it - kdec2_ks(ik,is);
                    %}
                    
                    % work decision
                    [Vh_ks(ik,is), decisn_ks(ik,is)] = max([Vh1 Vh2]);
                end % nk
            end % ns
            kdec1(:,:,ih) = kdec1_ks; kdec2(:,:,ih) = kdec2_ks;
            Vh(:,:,ih) = Vh_ks; decisn(:,:,ih) = decisn_ks;
            cdec1(:,:,ih) = cdec1_ks; cdec2(:,:,ih) = cdec2_ks;    
        end % nh
        % value function (average over h)
        Zt(:,:,it) = hp(1)*Vh(:,:,1);
        for iih = 2:nh
            Zt(:,:,it) = Zt(:,:,it) + hp(iih)*Vh(:,:,iih);
        end
        EZt(:,:,it) = Zt(:,:,it)*sprob';
        
        % labor supply decision (0 or 1)
        ndec = (decisn == 2);
        % capital decision
        kdec = (1-ndec).*kdec1 + ndec.*kdec2;
        % consumption
        cdec = (1-ndec).*cdec1 + ndec.*cdec2;
        
        ndect(:,:,it) = hp(1)*ndec(:,:,1);
        kdect(:,:,it) = hp(1)*kdec(:,:,1);
        cdect(:,:,it) = hp(1)*cdec(:,:,1);
        for iih = 2:nh
            ndect(:,:,it) = ndect(:,:,it) + hp(iih)*ndec(:,:,iih);
            kdect(:,:,it) = kdect(:,:,it) + hp(iih)*kdec(:,:,iih);
            cdect(:,:,it) = cdect(:,:,it) + hp(iih)*cdec(:,:,iih);
        end        
    end % nt
    
    % Simulation of the distribution function
    for it=2:nt-1
        gkt(:,:,it)=zeros(nk,ns);
        for is = 1:ns
            for ik = 1:nk
                % for each kgrid(ik), set the next period capital decision
                % function k1(k0,is)
                k0 = kgrid(ik);
                if k0 <= kgrid(1)
                    k1 = kdect(1,is,it-1);
                elseif k0 >= kgrid(nk)
                    k1 = kdect(nk,is,it-1);
                else
                    k1 = qinterp1(kgrid,kdect(:,is,it-1),k0);
                end
                % given k1(k0,is), gk is the new density. gk0 is the
                % previous density. For k0=kgrid(ik), we determine the cases
                % where gk(k1(k0,is),isp)
                if k1 <= kgrid(1)
                    gkt(1,:,it) = gkt(1,:,it) + gkt(ik,is,it-1)*sprob(is,:);
                elseif k1 >= kgrid(nk)
                    gkt(nk,:,it) = gkt(nk,:,it) + gkt(ik,is,it-1)*sprob(is,:);
                else
                    j=sum(kgrid<=k1)+1;
                    n0=(k1-kgrid(j-1))/(kgrid(j)-kgrid(j-1));
                    gkt(j,:,it) = gkt(j,:,it)    + n0*gkt(ik,is,it-1)*sprob(is,:);
                    gkt(j-1,:,it)= gkt(j-1,:,it) + (1-n0)*gkt(ik,is,it-1)*sprob(is,:);
                end
            end % ik
        end	% is
        gkt(:,:,it)=gkt(:,:,it)/sum(sum(gkt(:,:,it)));
        
        % pkt: price of real profit share
        KK=gkt(:,:,it)'*kgrid;
        pkt(it-1) = sum(KK);
        
        % Et: aggregate labor
        Et(it)=sum(sum(gkt(:,:,it).*ndect(:,:,it)));
        
        % Lt: aggregate effective labor
        smatrix=repmat(sgrid',nk,1);
        heff=smatrix.*ndect(:,:,it);
        Lt(it)=sum(sum(gkt(:,:,it).*heff)); 
        
        % Xt: aggregate intermediate input
        Xmu = (smatrix.^(1/mu)).*ndect(:,:,it);
        Xtnew(it) = (sum(sum(gkt(:,:,it).*Xmu)))^(mu);
        
        % Rkt: real profit
        Rkt(it) = alpha*At(it)*(K/Xtnew(it))^(alpha)*Xtnew(it);
        % rt: real interest rate 
        if it > 2
            rtnew(it-1) = (pkt(it-1)-pkt(it-2))/pkt(it-2) + Rkt(it-1)/pkt(it-2) ;
        end
    end % it
    
    rcrit = max(abs(rtnew-rt)./rtnew);
    Xcrit = max(abs(Xtnew-Xt)./Xtnew);
    crit = max([rcrit Xcrit]);
    
    q=q+1; 
    toc
    disp(['iteration: ' num2str(q)])
    disp(['max percentage error in (r,X): ' num2str(100*crit) '%'])
    
    rt = rt + psi0*(rtnew-rt);
    Xt = Xt + psi0*(Xtnew-Xt);
    
    save('TransfersMainLucas')    
end % q
