function [gk,K]=pdf_ke(kgrid,kdec,kgridnew,EP,ep)
% [gk,K] = pdf_ke(kgrid,kdec,kgridnew,EP,ep)
% < Heterogeneous Model with Idiosyncratic Shocks >
% Computation of the stationary PDF (k,e) by
% iteration on the density (PDF)
%
% Input:
% kgrid: grid of capital applied in the original value function
% kdec: matrix with dim (nk,ne) for the capital decision next period
%       k_value = kdec(k_index,e_index) IMPORTANT!!!!
% kgridnew: grid of capital to apply (may be finer than kgrid)
% EP: transition probability of e (EP[i,j]= Prob(t+1=j | t=i))
% ep: stationary distribution from EP (row vector such that ep*EP=ep) 
% 
% Output:
% gk: invariant distribution (pdf) of wealth and shocks
% K: aggregate capital
%
% Reference: Chapter 5 of "DGE Modeling," Heer and Maussner (2005)
% Last Modified: Jul 3, 2010


[nk,ne]=size(kdec);
nknew=length(kgridnew);

% initialization of the distribution functions
gk=ones(nknew,ne)/(nknew*ne); % for each density
gk=gk.*repmat(ep,nknew,1);

% iteration to find invariant distribution 
q=0; kritg=1; 
% iteration setup
maxit=500; tol=1e-20; 

% "computation of invariant distribution of wealth..";
while (q<=maxit) && (kritg>=tol)
    q=q+1; 
    gk0=gk;
    gk=zeros(nknew,ne);
    for ie=1:ne
        for ik=1:nknew
            k0 = kgridnew(ik);
            % for each kgridnew k0, designate the capital decision k1(k0,ie)
            if k0 <= kgrid(1) 
                k1 = kdec(1,ie); 
            elseif k0 >= kgrid(end);
                k1 = kdec(end,ie);
            else    
                k1 = qinterp1(kgrid,kdec(:,ie),k0);
            end
            % using k1(k0,ie), compute the density gk(k1(k0,ie),iie)
            % if k1 is not one of the gridpoints of kgridnew, give weights
            % to the gridpoints of kgridnew that are close to k1.
            if k1 <= kgrid(1);
                gk(1,:)=gk(1,:)+gk0(ik,ie)*EP(ie,:);
            elseif k1 >= kgrid(end);
                gk(end,:)=gk(end,:)+gk0(ik,ie)*EP(ie,:);
            else
                j=sum((kgridnew <= k1))+1; % determine the closest gridpoint above k1 and call it j
                n0=(k1-kgridnew(j-1))/(kgridnew(j)-kgridnew(j-1)); % weight on grid j
                % divide the values to gk(j,:) and gk(j-1,:). 
                gk(j,:)  = gk(j,:)  + n0*gk0(ik,ie)*EP(ie,:); 
                gk(j-1,:)= gk(j-1,:)+ (1-n0)*gk0(ik,ie)*EP(ie,:);
            end
        end
    end
    gk=gk/sum(sum(gk)); % why normalize if we started from a density? check if this is necessary...
    kritg=sum(sum(abs(gk0-gk))); 
end

% aggregate capital computation
kk1 = gk'*kgridnew;
K = sum(kk1);
