disp('Starting FullModel Script')

%% Check if we are running inside a batch script:
% if so, we take experiment variable from outer script.
% otherwise, this is the main script and we set experiment here
s = dbstack(1);
if isempty(s)
     clear all; close all; clc;

     global modelOptions;
     
    % choose Policy Experiment here
    %{
     experiments:
     experiment = 0 -> full model
     experiment = 1 -> all stabilizers reduced
     experiment = 2 -> proportional taxes reduced
     experiment = 3 -> flat tax
     experiment = 4 -> low transfers
     experiment = 6 -> income tax cut
    
    %}
    modelOptions.experiment = 0;
    
    %Choose Fiscal Adjustment Rule Here
    % 3 -> lump-sum tax adjusts to debt
    % 4 -> mixed Fiscal adjusment based on Leeper estimates
    % 5 -> balanced budget with LST adjusting
    % 6 -> distortionary taxes adjust.  no spending response.
    modelOptions.FiscalAdj = 4;
    
    
    %choose optional search effort extension
    modelOptions.search = false;
    
    %sticky prices  Don't change this, see robustness 12 below
    modelOptions.stickyPrices = true;
    
    %robustness
    % 0 -> baseline model
    % 1 -> low labor supply elasticity
    % 2 -> high labor supply elasticity
    % 3 -> Schmitt-Grohe-Uribe monetary policy rule
    % 6 -> GHH
    % 7 -> More aggressive monetary policy
    % 10 -> Less aggressive monetary policy
    % 12 -> flexible prices
    % 14 -> monetary policy rule with output term
    modelOptions.robustness = 0
    
else
    global modelOptions %#ok
    modelOptions.experiment = exp_in;
    modelOptions.FiscalAdj = fa_in;
    modelOptions.search = search_in;
    modelOptions.stickyPrices = sticky_in;
    modelOptions.robustness = robust_in
    clear exp_in fa_in endogUnemp_in sticky_in robust_in;
end


% Some options about the solution algorthim
modelOptions.IRF_Length = 100;
modelOptions.sim_T = 10000;


if modelOptions.search
    modelOptions.do_exact = true;
    modelOptions.do_mpa = false;
else
    modelOptions.do_exact = true;
    modelOptions.do_mpa = false;
end

%set the baseline parameters
if ~modelOptions.search && modelOptions.robustness ~= 6
    run('FullModel/parameters/initparams');
elseif modelOptions.search
    run('FullModel/parameters/initparams_search');
elseif modelOptions.robustness == 6
    run('FullModel/parameters/initparams_GHH');
end

%modify the parameters as per the experiments
if modelOptions.experiment > 0
    fname = ['FullModel/parameters/exper_' num2str(modelOptions.experiment)];
    run(fname);
    clear fname;
end
%modify the parameters as per the robustness exercise
modelOptions.GHH = false;
if modelOptions.robustness > 0
    fname = ['FullModel/parameters/robustness_' num2str(modelOptions.robustness)];
    run(fname);
    clear fname;
end




Params.warned = false;
[Params.parstart savePar] = guessHouseholdPolicyRules(~modelOptions.GHH);

if savePar
    par0 = Params.parstart; %#ok
    save([Params.dirpath '/initCond/par_ex' num2str(modelOptions.experiment) ...
        '_se' num2str(modelOptions.search) ...
        '_rob' num2str(modelOptions.robustness) '.mat'], 'par0');
    clear par0;
end



% it is crucial that skill = 0 if not employed
assert(all(Params.skill(~Params.employed) == 0))

%% COMPUTE STEADY STATE:


EntIncomeRelativeToMean = Params.incomeshare*(1+Params.nu);
if modelOptions.experiment == 2
    xe_stst = EntIncomeRelativeToMean + 0.564;
else
    xe_stst = EntIncomeRelativeToMean;
end
xe_stst = broyden(@compute_stst,xe_stst);

[xe_resid, KLequ, xaggstst, par, D, Env, X] = compute_stst(xe_stst);

assert(abs(xe_resid) < 1e-6, ['xe_resid is large = ' num2str(xe_resid)]);

if savePar
    par0 = Params.parstart; %#ok
    save([Params.dirpath '/initCond/par_ex' num2str(modelOptions.experiment) ...
        '_se' num2str(modelOptions.search) ...
        '_rob' num2str(modelOptions.robustness) '.mat'], 'par0');
    clear par0;
end

if modelOptions.experiment == 3 || modelOptions.experiment == 1
    disp(['lump-sum tax is ' num2str(xaggstst(strcmp(Params.aggnames,'LumpSumTax')))])
    disp('It should be zero')
end


%% CHECK EULER RESIDUALS IN STEADY STATE:

if true
    nxtest = 1000;
    xx2 = linspace(Params.xmin,Params.xmax,nxtest)';

    [margtaxe, taxpaide] = interp_tax(xe_stst, Params.incometax);
    [ z, M, wage, ii, r, rhat] = SteadyStatePrices( margtaxe );

    laborMarketStatus = 0;
    ppinext = 1;
    taxshock = 0;
    dtau = 0;
    
    resC = eulerres(par,par,1+ii,1+ii,wage,wage,laborMarketStatus,ppinext,taxshock,taxshock,dtau,dtau,true,xx2);
    ndec = 3;
    
    
    resC = reshape(resC,ndec*nxtest,Params.npp);

    nresplot = ndec*Params.npp;
    for ip=1:nresplot
        subplot(Params.npp,ndec,ip);
        plot(xx2,resC(1+nxtest*(ip-1):nxtest*ip));
        %ylim([-5e-4 5e-4]);

    end

end

%% Analyze the steady state

%open file for writing
fname = [Params.dirpath '/results/steady_state_experiment_ex' num2str(modelOptions.experiment) ...
    '_se' num2str(modelOptions.search) '_rob' num2str(modelOptions.robustness) ...
    '_sticky' num2str(modelOptions.stickyPrices) '.txt'];
fid= fopen(fname,'w');



% ***compute statistics of wealth distribution***

%household wealth is just bond wealth
b = Params.knotDistrK;
b = repmat(b,Params.npp,1);

pop = Params.nu * D;  % this is the mass of households in each wealth cell

%capital-owner wealth is 
wealthCO = xaggstst(strcmp(Params.aggnames,'be')) + xaggstst(strcmp(Params.aggnames,'K'));


[giniIndex, Lorenz] = gini([pop; 1], [b; wealthCO], false);



dispAndSave(['Gini coefficient is ' num2str(giniIndex)], fid);
dispAndSave('Wealth quintiles:', fid)
dispAndSave('quintile    share of wealth', fid)
dispAndSave(['1           ' num2str(...
    Lorenz(find(Lorenz(:,1) <= 0.2,1,'last'),2) - 0 ...
    )], fid)
dispAndSave(['2           ' num2str(...
    Lorenz((find(Lorenz(:,1) <= 0.4,1,'last')),2) - Lorenz(find(Lorenz(:,1) <= 0.2,1,'last'),2)...
    )], fid)
dispAndSave(['3           ' num2str(...
    Lorenz(find(Lorenz(:,1) <= 0.6,1,'last'),2) - Lorenz(find(Lorenz(:,1) <= 0.4,1,'last'),2)...
    )], fid)
dispAndSave(['4           ' num2str(...
    Lorenz(find(Lorenz(:,1) <= 0.8,1,'last'),2) - Lorenz(find(Lorenz(:,1) <= 0.6,1,'last'),2)...
    )], fid)
dispAndSave(['5           ' num2str(...
    1 - Lorenz(find(Lorenz(:,1) <= 0.8,1,'last'),2)...
    )], fid)



dispAndSave(' ',fid);
dispAndSave('Fraction of total population with low wealth', fid);
dispAndSave('income type     share of pop.       fraction with low weatlh ',fid);
[ ~, ~, wage, ii, ~, ~, ~] = SteadyStatePrices( margtaxe );
for ip = 1:Params.npp
    lowWealth = average_income(D,par,mod(ip-1,3)+1, wage, ii);
    [lwshare, popshare] =expect_low_wealth(D,ip,lowWealth);
    dispAndSave([num2str(ip) '           ' num2str(popshare*Params.nu/(1+Params.nu)) '             ' num2str(lwshare)], fid)
end


%compute the MPC table
tmp = reshape(D,Params.ndstst,Params.npp);
tmp = tmp./(ones(Params.ndstst,1)*sum(tmp));
tmp = cumsum(tmp);

trans = 0.005;

dispAndSave('Computing marginal propensities to consume',fid);
dispAndSave('Income     Wealth percentile:',fid);
dispAndSave('type:      10th              25th              50th',fid);
for ip = 1:Params.npp
    bi = find(tmp(:,ip) >= 0.1,1,'first');
    b = Params.knotDistrK(bi);
    [m10, check] = mpc(b,ip, trans, Params.parstart, Params.parstart,1+ii,wage);
    assert(abs(check) < 1e-10)
    
    bi = find(tmp(:,ip) >= 0.25,1,'first');
    b = Params.knotDistrK(bi);
    [m25, check] = mpc(b,ip, trans, Params.parstart, Params.parstart,1+ii,wage);
    assert(abs(check) < 1e-10)
    
    bi = find(tmp(:,ip) >= 0.5,1,'first');
    b = Params.knotDistrK(bi);
    [m50, check] = mpc(b,ip, trans, Params.parstart, Params.parstart,1+ii,wage);
    assert(abs(check) < 1e-10)
    
    dispAndSave([num2str(ip) '           ' num2str(m10(1))  '           ' num2str(m25(1)) '           ' num2str(m50(1))],fid);
    
end

clear b bi m10 m25 m50 ip trans tmp check;


% close steady state results file.
fclose(fid);

%% package some steady state variables to be saved

%prepareZLB;

%save the distribution of wealth
stst_soln.D = D;

%compute: income and wealth for each component of D and entrepreneur
stst_soln.income = zeros(Params.ndstst,Params.npp);

par = par2wide(par);

for ip=1:Params.npp
      S = savingspline(par(Params.par_sind,ip));
      N = nspline(par(Params.par_nind,ip));
    
      %we have end of period assets in last period and this period,
      %and we have to figure out this period's consumption:
      xthis = Params.knotDistrK;
      sthis = interp_savspline(S,xthis);
      [~, ~, ~, ~, stst_soln.income(:,ip)] = get_cnt(xthis,sthis,N,1+ii,wage,ip);
end

stst_soln.wealth = Params.knotDistrK;

stst_soln.xe = xaggstst(strcmp(Params.aggnames,'xe'));
stst_soln.wealthe = xaggstst(strcmp(Params.aggnames,'be')) ...
    + xaggstst(strcmp(Params.aggnames,'K')) ...
    + xaggstst(strcmp(Params.aggnames,'firmValue'));

stst_soln.Y = xaggstst(strcmp(Params.aggnames,'Y'));


%% Linearization


Params.ntotal = length(X);

% the number of expectational errors
if modelOptions.search
    nsearch = sum(~Params.employed)*Params.nn;
    neta = (Params.nc+Params.nv)*Params.npp + nsearch + Params.nAggEta;  
else
    neta = Params.nc*Params.npp + Params.nAggEta;
end

[X2,varindx] = setix(X,neta,Params.nz); 

Env.KLstst = KLequ;
Env.varix = varindx;
[residStst,Env] = equ(X2,Env);
if any(abs(residStst) >1e-6)
    error('wrong stst');
end
  

njac = 20*ones(size(Env.varix.x));
njac(Env.iVarStatic)=500;
Env = dojacob(@equ,X2,Env,njac,Env);

%% Solution Linear RatEx Model -- first exactly


FullModel_exact_sol;




%% compute a few things

get_calibration_moments;

%% save results 


fname = [Params.dirpath '/results/experiment_' num2str(modelOptions.experiment) '_FiscalAdj_' num2str(modelOptions.FiscalAdj) '_search_' num2str(modelOptions.search)  '_sticky_' num2str(modelOptions.stickyPrices) '_robustness_' num2str(modelOptions.robustness) '.mat'];

save_vars = {'stst_soln'};
if modelOptions.do_exact
    save_vars = [save_vars 'exact_soln'];
end

if modelOptions.do_mpa
    save_vars = [save_vars 'MPA_soln'];
end

save(fname,save_vars{:});



%% Prepare for transitions
prepare_transition(Env,par,D);

