%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FUNCTION THAT DEFINES EULER AND LABOR SUPPLY RESIDUALS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% inputs:
%   par:           parameter vector for_ current savings function_
%   parnext:       parameter vector for_ next periods' savings function_
%   R,wage: 1+interest,wage paid this period.  i.e interest on assets entering the period
%   Rnext,wagenext: 1+interest,wage paid next period
%   steadyState:   T/F are we computing the steady state?
%   xthis:        grid at which_ to compute Euler residuals; default: knot points of spline
% output:
function [res, accuracy_cthis, accuracy_MUexp, accuracy_sthis] = eulerres(par,parnext,R,Rnext,wage,wagenext,laborMarketStatus,ppinext,taxshock,taxshocknext,dtau,dtaunext,steadyState,xthis)

global Params modelOptions;


npp = Params.npp;

par = par2wide(par);
parnext = par2wide(parnext);


%figure out if we are doing a test of residuals off the grid
IsATest = exist('xthis','var');

if ~IsATest
    nc = Params.nc;
    nv = Params.nv;
    nn = Params.nn;
else
    nc = length(xthis);
    nv = nc;
    nn = nc;
end

if nargout > 1
    accuracy_cthis = zeros(nc,npp);
    accuracy_MUexp = zeros(nc,npp);
end

for ip=1:npp
    
    S = savingspline(par(Params.par_sind,ip));
    N = nspline(par(Params.par_nind,ip));
    V = vspline(par(Params.par_vind,ip));
    
    
    if IsATest % xthis given
        sthis = interp_savspline(S,xthis);
    else % take x as starting value of assets
        xthis = S.x(1:end-1);  % last point in S is for interpolation
        sthis = S.y(1:end-1);
    end
    
    
    
    %we have end of period assets in last period and this period,
    %and we have to figure out this period's consumption:
    [cthis, nthis] = get_cnt(xthis,sthis,N,R,wage,ip,taxshock,dtau);
    
    
    
    if(any(cthis<0))  %signal inadmissible value to routine 'broydn';
        %         disp('A');
        %         ip
        %         I = find(cthis<0)
        %         nthis(I)
        %         xthis(I)
        %         sthis(I)
        res = 1e100;
        return;
    end
    
    
    if(ip==1) % initialize z:
        res = initsize(zeros(nc+nn+nv,Params.npp),par,parnext,R,Rnext,wage,wagenext,xthis,ppinext,taxshock,taxshocknext,dtau,dtaunext);
    end
    
    
    assets = sthis;
    
    MUexp = 0;
    
    
    for jp=1:npp
        
        pp = transProb(ip,jp,laborMarketStatus, nthis);
        
        
        if(any(pp>0))
            Sn = savingspline(parnext(Params.par_sind,jp));
            Nn = nspline(parnext(Params.par_nind,jp));
            
            
            snext = interp_savspline(Sn,assets);
            
            
            [cnext, nnext, margtaxnext] = get_cnt(assets,snext,Nn,Rnext,wagenext,jp,taxshocknext,dtaunext);
            
            if(any(cnext<1e-8))  %signal inadmissible value to routine 'broydn';
                %           disp('B');
                res = 1e100;
                return;
            end
            
            if modelOptions.GHH
                MUnext = margutilC_GHH(cnext,nnext);
            else
                MUnext = margutilC(cnext);
            end
            
            MUexp = MUexp + pp.*MUnext .* (1+(1-margtaxnext)*(Rnext-1))/ppinext*(1+Params.tauC+dtau)/(1+Params.tauC+dtaunext);
            
            
        end
    end
    
    if nargout > 1
        accuracy_cthis(:,ip) = cthis;
        accuracy_MUexp(:,ip) = MUexp;
        accuracy_sthis(:,ip) = sthis;
    end
    
    % Euler residual:
    % expressed in relative consumption units:
    %notice that R is incorporated into MUexp
    if modelOptions.GHH
        res(nv+nn+1:end,ip) = 1 - invmargutilC(Params.betah*MUexp)./(cthis - Params.psi1*nthis.^(1+Params.psi2)/(1+Params.psi2));
    else
        res(nv+nn+1:end,ip) = 1 - invmargutilC(Params.betah*MUexp)./cthis;
    end
    
    
    %calculations for labor supply residuals
    clear nthis sthis cthis;
    
    if IsATest % xthis given
        nthis1 = interp_nspline(N,xthis,false);
    else % take x as starting value of assets
        xthis = N.x(1:end-1);  % last point in N is for interpolation
        nthis1 = N.y(1:end-1);
    end
    
    sthis = interp_savspline(S,xthis);
    assets = sthis/ppinext;
    
    
    
    if Params.estatus(ip) == 1
        
        [cthis, ~, margTaxthis] = get_cnt(xthis,sthis,N,R,wage,ip,taxshock,dtau);
        
        if(any(cthis<0))  %signal inadmissible value to routine 'broydn';
            %           disp('C');
            res = 1e100;
            return;
        end
        
        if modelOptions.GHH
            laborResid = (1+Params.tauC+dtau) .* Params.psi1 .* nthis1.^Params.psi2   ./(Params.skill(ip)*wage.*(1-margTaxthis))-1;  %notice that n is allowed to be negative only here
        else
            laborResid = (1+Params.tauC+dtau) .* Params.psi1 .* nthis1.^Params.psi2   ./(cthis.^(-Params.sigma).*Params.skill(ip)*wage.*(1-margTaxthis))-1;  %notice that n is allowed to be negative only here
        end
        
        
    elseif modelOptions.search
        
        dValue = 0;
        
        for jp=1:npp
            %loop through next period states
            
            [~, dprobdn] = transProb(ip,jp,laborMarketStatus,nthis1);
            
            if any(dprobdn ~= 0)
                Vn = vspline(parnext(1:Params.nv,jp));
                Vnext = interp_vspline(Vn,assets);
                dValue = dValue + dprobdn .*Vnext;
            end
        end
        
        laborResid = Params.psi1 * nthis1.^Params.psi2 - Params.betah * dValue;
        
        
    end
    
    %store results
    if Params.estatus(ip) == 1 || modelOptions.search
        res(nv+1:nv+nn,ip) = laborResid;
    end
    
    clear nthis sthis cthis;
    
    
    % ---------- calculations for V residuals --------------
    if modelOptions.search || steadyState
        
        
        if IsATest % xthis given
            Vthis = interp_vspline(V,xthis);
        else
            Vthis = V.y(1:end-1);
            xthis_chv = V.x(1:end-1);
            xthis = exp(xthis_chv)-1;  %change of variables
        end
        sthis = interp_savspline(S,xthis);
        
        [cthis, nthis] = get_cnt(xthis,sthis,N,R,wage,ip,taxshock,dtau);
        assert(all(cthis > 0), 'error in eulerres: cthis is zero or negative.')
        
        
        %build the expected continuatiuon value
        assets = sthis/ppinext;
        Vexp = 0;
        for jp=1:npp
            
            pp = transProb(ip,jp,laborMarketStatus,nthis);
            
            if any(pp>0)
                Vn = vspline(parnext(1:Params.nv,jp));
                Vnext = interp_vspline(Vn,assets);
                Vexp = Vexp + pp.*Vnext;
            end
        end
        
        if modelOptions.GHH
            res(1:nv,ip) = -Vthis +  log(cthis - Params.psi1*(nthis.^(1+Params.psi2))/(1+Params.psi2)) + Params.betah*Vexp;
        else
            res(1:nv,ip) = -Vthis +  log(cthis) - Params.psi1*(nthis.^(1+Params.psi2))/(1+Params.psi2) + Params.betah*Vexp;
        end
        
    end
    
end  %close loop over this period's income state


if modelOptions.search || (IsATest && steadyState)
    
    res = [reshape(res(1:nv,:),nv*Params.npp,1);
        reshape(res(nv+1:nv+nn,:),nn*Params.npp,1);
        reshape(res(nv+nn+1:end,:),nc*Params.npp,1)];
elseif steadyState
    res = [reshape(res(1:nv,:),nv*Params.npp,1);
        reshape(res(nv+1:nv+nn,Params.employed), nn*Params.npp_n,1);
        reshape(res(nv+nn+1:nv+nn+nc,:), nc*Params.npp,1)];
else
    res = [ reshape(res(nv+1:nv+nn,Params.employed), nn*Params.npp_n,1);
        reshape(res(nv+nn+1:nv+nn+nc,:), nc*Params.npp,1)];
end


if nargout > 1
    accuracy_cthis = accuracy_cthis(:);
    accuracy_MUexp = accuracy_MUexp(:);
end