%Check if we are running inside a batch script:
% if so, we take experiment variable from outer script.
% otherwise, this is the main script and we set experiment here
s = dbstack(1);
if isempty(s)
     clear all; close all; clc;

    global modelOptions;
     
    % choose Policy Experiment here
    % experiments:
    % experiment = 0 -> full model
    % experiment = 1 -> cut all stabilizers
    % experiment = 2 -> simplified taxes
    % experiment = 3 -> proportional tax
    % experiment = 4 -> low transfers
    % experiment = 6 -> income tax cut
    % experiment = 7 -> no government
    modelOptions.experiment = 0;
    
    %Choose Fiscal Adjustment Rule Here
    % 3 -> lump-sum tax adjusts to debt
    % 4 -> mixed Fiscal adjusment based on Leeper estimates
    % 5 -> balanced budget with LST adjusting
	modelOptions.FiscalAdj = 4;

    
    
    % choose sticky prices or flexible prices
    modelOptions.stickyPrices = true;
    
    %robustness
    modelOptions.robustness = 0

    
else
    global modelOptions %#ok
    modelOptions.experiment = exp_in;
    modelOptions.FiscalAdj = fa_in;
    modelOptions.stickyPrices = sticky_in;
    modelOptions.robustness = robust_in
    clear exp_in fa_in endogUnemp_in sticky_in robust_in;
end


initParams;

if modelOptions.experiment > 0  && modelOptions.experiment ~= 3
    fname = ['FullModel/parameters/exper_' num2str(modelOptions.experiment)];
    run(fname);
    clear fname;
elseif modelOptions.experiment == 3
    Params.incometax = 0.1224;
end


if modelOptions.robustness > 0
    fname = ['FullModel/parameters/robustness_' num2str(modelOptions.robustness)];
    run(fname);
    clear fname;
end


%% Solve for steady state

x = broyden(@solveForSS,[1+Params.nu;0.3]);


[xresid Env] = solveForSS(x);
assert(all(abs(xresid) < 1e-5));


%% Checks

if modelOptions.experiment == 3
    disp('----')
    disp(['steady state lump sum tax is ' num2str(Env.stst(strcmp(Params.names,'LumpSumTax')))]);
    disp('it should be zero.  See initparams.m')
    disp('----')
end

%% Dynamics

Env.nx = length(Params.names);

Params.ix = [1:Env.nx];
Params.ixlag = [Env.nx+1:Env.nx*2];
if modelOptions.stickyPrices
    Params.ieta = Env.nx*2+1:Env.nx*2+5;
    Params.ieps = Env.nx*2+[6:6+Params.nz-1];
    X = zeros(Env.nx,1);
    [X2,varindx] = setix(X,5,Params.nz); 
else
    Params.ieta = Env.nx*2+1:Env.nx*2+3;
    Params.ieps = Env.nx*2+[4:4+Params.nz-1];
    X = zeros(Env.nx,1);
    [X2,varindx] = setix(X,3,Params.nz); 
end



Env.varix = varindx;
[residStst,Env] = equ(X2,Env);
if(max(abs(residStst))>1e-6)
    error('wrong stst');
end



%Env.test = true;
njac = 20*ones(size(Env.varix.x));
%njac(Env.iVarStatic)=500;
Env = dojacob(@equ,X2,Env,njac,Env);


%% Solution Linear RatEx Model -- first exactly

disp('Solving the model exactly')
[G1,impact,eu,abseig] = solveexact(Env);
if(~all(eu==1))  %signal failure:
  G1 = [];
  warning('sims failed');
end





%% Compute IRFs


myIRF_T = 60; %num periods to do in IRF



irf_z = [];
y = Params.SigmaEpsz *impact(:,1);
for i = 1:myIRF_T
    irf_z = [irf_z; y'];
    y = G1 * y;
end

if modelOptions.experiment ~= 7
    irf_G = [];
    y =  impact(:,2);
    for i = 1:myIRF_T
        irf_G = [irf_G; y'];
        y = G1 * y;
    end
end

if modelOptions.stickyPrices

    irf_mp = [];
    y = Params.SigmaEpsmp *impact(:,3);
    for i = 1:myIRF_T
        irf_mp = [irf_mp; y'];
        y = G1 * y;
    end

    irf_markup = [];
    y = Params.Sigmamarkup *impact(:,4);
    for i = 1:myIRF_T
        irf_markup = [irf_markup; y'];
        y = G1 * y;
    end
    


end

%% Make IRF plots

s = dbstack(1);
if isempty(s)  %this is a trick to see if we are running in batch mode (no IRFs in batch mode)

    vars = {'K', 'Y', 'G', 'B', 'C', 'x', 'ii', 'ppi', 'N', 'U', 'NLF', 'mpshock', 'z'};
    
    
    for sh = 1:Params.nz
        figure;
        if sh == 1
            irf = irf_z;
        elseif sh == 2
            irf = irf_G;
        elseif sh == 3
            irf = irf_mp;
        elseif sh == 4
            irf = irf_markup;
        end
        for v = 1:length(vars)
            iv = strcmp(Params.names,vars{v});
            if ~isempty(find(iv))
                subplot(ceil((length(vars)+1)/2),2,v);
                if ~any(strcmp({'ii', 'ppi', 'mpshock'}, vars{v}))  && sh ~= 2
                    logIRF = irf(:,iv)/Env.stst(iv);
                    plot(1:myIRF_T,logIRF);
                else
                    plot(1:myIRF_T,irf(:,iv));
                end
                title(vars{v});
            end
        end
        
        if modelOptions.stickyPrices
            ppi = irf(2:end,strcmp(Params.names,'ppi'));
        else
            ppi = 0;
        end
        subplot(ceil((length(vars)+1)/2),2,v+1);
        realRate = irf(1:end-1,strcmp(Params.names,'ii')) - ppi;
        plot(realRate);
        title('real interest rate');
    end
    
    
end

  
%% compute variances and other moments

compute_variances;


%% simulate to get average

Hagg = eye(length(impact));

sim_T = 10000;  

[simaggdat, shocks] = simulateSystem(G1,impact,Hagg,sim_T,Params.Sigma,Params.seed);
simaggdat = simaggdat + Env.stst * ones(1,sim_T);
simul_avg = mean(simaggdat,2);
simaggdat = simaggdat';
  



%% save results 

fname = ['RepAgent/results/experiment_' num2str(modelOptions.experiment) '_FiscalAdj_' num2str(modelOptions.FiscalAdj)  '_search_0_sticky_' num2str(modelOptions.stickyPrices) '_robustness_' num2str(modelOptions.robustness) '.mat'];
xaggstst = Env.stst;  Params.aggnames = Params.names;

if modelOptions.stickyPrices
    irf_agg = zeros(size(irf_z,1),size(irf_z,2),5);
    irf_agg(:,:,1) = irf_z;
    irf_agg(:,:,2) = irf_G;
    irf_agg(:,:,4) = irf_mp;
    irf_agg(:,:,5) = irf_markup;
else 
    irf_agg = irf_z;
end


save(fname,'xaggstst', 'irf_agg', 'V0', 'V1', 'Params','simul_avg', 'simaggdat', 'shocks');

