function [margTax, taxpaid,dmarg] = interp_tax(y, incometax, taxshock)
%[margTax, taxpaid,dmarg] = interp_tax(y, incometax, taxshock)
% takes a level of income or vector of incomes
% and an income tax system 
% produces marginal tax rates for each y and tax liability for each y
% It is assumed that mean income is 1 in steady state.
%
% An income tax system can take two forms:
% progressive tax:
% income tax is cubic up to a break point then flat
% this is coded in a structure with fields c (polynomial coefficients),
% ad (anti-derivative of coefficients), bp (break points), ybp (marginal
% rate at bp), medianincome (scale factor for income levels)
% Linear tax:
% incometax is the tax rate (a scalar)

if nargin < 3
    taxshock = 0;
end

if ~isstruct(incometax)
    %linear tax
    margTax = incometax*ones(size(y)) + taxshock;
    taxpaid = (incometax+taxshock)*y;
    dmarg = zeros(size(y));
else
    %progressive tax:
    
    c = incometax.marg;
    ad = incometax.cumul;
    bp = incometax.breakpoint;
    ybp = incometax.breakval;

    scaleFac = 1;

    x = y/scaleFac;



    ys = initsize(zeros(size(x)),y);
    I = x < bp;
    ys(I) = polyval(c, x(I));
    ys(~I) = ybp;

    margTax = ys + taxshock;
    
    if nargout < 2, return; end

    %accumulate tax paid like this
    taxpaid = initsize(zeros(size(x)),y);
    taxpaid(I) = polyval(ad,x(I));
    taxpaid(~I) = polyval(ad,bp) + (x(~I)-bp)*ybp;
    taxpaid = scaleFac*taxpaid;

    
    taxpaid = taxpaid + y*taxshock;
    if nargout < 3, return; end
    
    
    %derivative of marginal tax rate
    dmarg = initsize(zeros(size(x)),y);
    I = x < bp;
    dmarg(I) = polyval(polyder(c), x(I));
    dmarg(~I) = 0;
end

