% Analysis of sales and inventory data

%% Read in raw data

% Load data (monthly: first column = sales; second column = beginning of
% month inventory)
[data labels] = xlsread('SalesInventory.xlsx');

% Length of sample (months)
T = size(data,1);

% Extract sales and inventory series
sales = data(:,1);
inventory = data(:,2);

% Date of first observation
begindate = datevec(labels(2,1),'dd/mm/yyyy');

% Load housing stock data
housingstock = xlsread('HousingStock.xlsx');
housingstock = 1000*housingstock(:,2);

% Load fraction of homes not rented
notrented = xlsread('NotRented.xlsx');
notrented = (1/100)*notrented(:,2);
notrentedadjustment = false; % default for Figures 1-4, set to true to generate Figure 11

% Load real house price index and align with sales data
price = xlsread('RealHPI.xlsx');
price = [NaN*ones(T - size(price,1),1);price];

%% Seasonal adjustment

% Multiplicative seasonal adjustment
sales = seasonadj(sales);
inventory = seasonadj(inventory);

%% Adjustment to quarterly series

% Quarterly sample size (disgarding incomplete quarters at end of sample)
T = floor(T/3);

% Sum all sales within quarters
sales = sum(reshape(sales(1:(3*T),:),3,T))';

% Take beginning of first month inventory as beginning of quarter
inventory = reshape(inventory(1:(3*T),:),3,T)'*[1 0 0]';

% Average prices within quarters
price = mean(reshape(price(1:(3*T),:),3,T))';

%% Definitions and accounting identities

% Drop final observation because listings requires one future inventory
% observation
T = T - 1;

% Construct listings series from accounting identity
listings = inventory(2:T+1) - inventory(1:T) + sales(1:T);

% Construct houses for sale series
forsale = (inventory(1:T) + inventory(2:T+1))/2;

% Trim series
sales = sales(1:T);
inventory = inventory(1:T);
price = price(1:T);

%% Linear interpolation of housing stock to produce a quarterly series

% Number of years
N = size(housingstock,1);

% Linear interpolation
hstock = housingstock(1) + [0;cumsum(reshape((diff(housingstock)*((1/8)*ones(1,8)))',8*(N-1),1))];

%% Linear interpolation of fraction of homes not rented to produce a quarterly series

% Number of years
N = size(notrented,1);

% Linear interpolation
notrented = notrented(1) + [0;cumsum(reshape((diff(notrented)*((1/8)*ones(1,8)))',8*(N-1),1))];

%% Scale sales and inventories by fraction of homes not rented (optional)

if notrentedadjustment
    sales = notrented.*sales;
    listings = notrented.*listings;
    inventory = notrented.*inventory;
    forsale = notrented.*forsale;
end

%% Sales and listing rates, and growth rate of housing stock

% Sales rate
salesrate = sales./forsale;

% Listing rate
listingrate = listings./(hstock - forsale);

% Fraction for sale
fracforsale = forsale./hstock;

% Growth rate of housing stock
growthrate = diff(hstock)./hstock(1:end-1);

% Assume constant growth rate for initial data point
growthrate = [growthrate(1);growthrate];

%% Average rates and durations

% Average sales rate (annual)
avsalesrateannual = 4*mean(salesrate);

% Average listing rate (annual)
avlistingrateannual = 4*mean(listingrate);

% Average growth rate (annual)
avgrowthrateannual = 4*mean(growthrate);

% Average housing stock
avhstock = mean(hstock);

% Time to sell (years)
Ts = 1/avsalesrateannual;

% Time spent living in house (years)
Tn = 1/avlistingrateannual;

% Quarterly rates
avsalesrate = avsalesrateannual/4;
avlistingrate = avlistingrateannual/4;
avgrowthrate = avgrowthrateannual/4;

%% Construct quarterly date labels

% Beginning year and month
beginyear = begindate(1);
beginmonth = begindate(2);
beginqtr = 1;

% Advance month number to middle of first quarter
beginmonth = beginmonth + 1;

% Date vectors
dates = zeros(T,6);
dates(:,3) = ones(T,1);
dates(:,2) = mod((beginmonth:3:(beginmonth+3*(T-1)))',12);
dates(:,1) = beginyear + floor(((beginmonth:3:(beginmonth+3*(T-1)))' - dates(:,2))/12);

% Date labels
datelab = datenum(dates);

%% Changes over particular time periods

% Sales and listings adjusted for changes in total housing stock

% 1995-2003 period
t0 = ( 4*(1994 - beginyear) + ( 1 - beginqtr ) ) + 1;
t = ( 4*(2003 - beginyear) + ( 4 - beginqtr ) ) + 1;
t0 = (t0:t0+3)';
t = (t-3:t)';
chgsales9503 = 100*(mean(log(sales(t))) - mean(log(sales(t0))));
chglistings9503 = 100*(mean(log(listings(t))) - mean(log(listings(t0))));
chgsalesrate9503 = 100*(mean(log(salesrate(t))) - mean(log(salesrate(t0))));
chglistingrate9503 = 100*(mean(log(listingrate(t))) - mean(log(listingrate(t0))));
chgfracforsale9503 = 100*(mean(log(fracforsale(t))) - mean(log(fracforsale(t0))));
chgforsale9503 = 100*(mean(log(forsale(t))) - mean(log(forsale(t0))));
chgstock9503 = 100*(mean(log(hstock(t))) - mean(log(hstock(t0))));
chgsalesadjstock9503 = chgsales9503 - chgstock9503;
chglistingsadjstock9503 = chglistings9503 - chgstock9503;
chgprice9503 = 100*(mean(log(price(t))) - mean(log(price(t0))));

% 1995-2006 period
t0 = ( 4*(1995 - beginyear) + ( 1 - beginqtr ) ) + 1;
t = ( 4*(2006 - beginyear) + ( 4 - beginqtr ) ) + 1;
t0 = (t0:t0+3)';
t = (t-3:t)';
chgsalesadjstock9506 = 100*(mean(log(sales(t))) - mean(log(sales(t0)))) - 100*(mean(log(hstock(t))) - mean(log(hstock(t0))));
chgprice9506 = 100*(mean(log(price(t))) - mean(log(price(t0))));

% 2007-2009 period
t0 = ( 4*(2007 - beginyear) + ( 1 - beginqtr ) ) + 1;
t = ( 4*(2009 - beginyear) + ( 4 - beginqtr ) ) + 1;
t0 = (t0:t0+3)';
t = (t-3:t)';
chgsalesadjstock0709 = 100*(mean(log(sales(t))) - mean(log(sales(t0)))) - 100*(mean(log(hstock(t))) - mean(log(hstock(t0))));
chgprice0709 = 100*(mean(log(price(t))) - mean(log(price(t0))));

%% Figure 1: Housing-market activity

% Compute log differences of sales, listings, for sale stock, and total stock relative to initial values
fig1a = 100*[(log(sales) - log(sales(1))) (log(listings) - log(listings(1))) (log(forsale) - log(forsale(1))) (log(hstock) - log(hstock(1)))];

% Plot figure
varlab = {'Transactions';'Listings';'For sale';'Stock'};
figure;
subplot(2,1,1);
plot(datelab,fig1a);
datetick('x','yyyy','keeplimits');
legend(varlab);

% Compute log differences of sales and listing rates and for sale fraction relative to initial values
fig1b = 100*[(log(salesrate) - log(salesrate(1))) (log(listingrate) - log(listingrate(1))) (log(fracforsale) - log(fracforsale(1)))];

% Plot figure
varlab = {'Sales rate';'Listing rate';'For sale %'};
subplot(2,1,2);
plot(datelab,fig1b);
datetick('x','yyyy','keeplimits');
legend(varlab);

%% Figure 2: Counterfactuals for transactions using steady-state houses for
%% sale

% Sales volume with all rates varying
cfsssales = (salesrate.*listingrate.*hstock)./(salesrate + listingrate + growthrate);

% Sales volume with varying sales rate
cfsssalesvars = (avlistingrate*avhstock*salesrate)./(salesrate + avlistingrate);

% Sales volume with varying listing rate
cfsssalesvarn = (avsalesrate*avhstock*listingrate)./(avsalesrate + listingrate);

% Sales volume with varying housing stock
cfsssalesvarg = (avsalesrate*avlistingrate.*hstock)./(avsalesrate + avlistingrate + growthrate);

% Log differences relative to initial value of each series
fig2 = 100*[(log(cfsssales) - log(cfsssales(1))) (log(cfsssalesvars) - log(cfsssalesvars(1))) ... 
    (log(cfsssalesvarn) - log(cfsssalesvarn(1))) (log(cfsssalesvarg) - log(cfsssalesvarg(1)))];

% Plot figure
varlab = {'Vary sales rate';'Vary listing rate';'Vary stock';'Vary all three'};
figure;
plot(datelab,fig2);
datetick('x','yyyy','keeplimits');
legend(varlab);

%% Figure 3: Actual and counterfactual transactions out of steady state

% Vary the sales rate
cfsalesvars = cfsales(salesrate,avlistingrate*ones(T,1),zeros(T,1),avhstock*ones(T,1));

% Vary the listing rate
cfsalesvarn = cfsales(avsalesrate*ones(T,1),listingrate,zeros(T,1),avhstock*ones(T,1));

% Vary the housing stock
cfsalesvarg = cfsales(avsalesrate*ones(T,1),avlistingrate*ones(T,1),growthrate,hstock);

% Log differences relative to initial value of sales series
fig3 = 100*[(log(sales) - log(sales(1))) (log(cfsalesvars) - log(cfsalesvars(1))) ... 
    (log(cfsalesvarn) - log(cfsalesvarn(1))) (log(cfsalesvarg) - log(cfsalesvarg(1))) ];

% Plot figure
varlab = {'Vary sales rate';'Vary listing rate';'Vary stock';'Data'};
figure;
plot(datelab,fig3);
datetick('x','yyyy','keeplimits');
legend(varlab);

%% Series for withdrawals and relistings

% Eventual withdrawals series derived from projection on sales rate
meaneventualwithdraw = 0.42;
regcoeffsalesrate = -0.6;
eventualwithdraw = meaneventualwithdraw*(1 + regcoeffsalesrate*(salesrate - mean(salesrate))/mean(salesrate));

% Fraction ever relisted (assumed constant) 
relistever = 0.5;

% Implied moving rate
movingrate = (1/2)*( (1 + (1-relistever).*(eventualwithdraw./(1-eventualwithdraw)).*(1./(1 + (growthrate./salesrate))) ...
    - (1-relistever).*(growthrate./listingrate) ) + sqrt( (1 + (1-relistever).*(eventualwithdraw./(1-eventualwithdraw)).*(1./(1 ... 
    + (growthrate./salesrate))) - (1-relistever).*(growthrate./listingrate) ).^2 + 4*(1-relistever).*(growthrate./listingrate).*( 1 ...
    + (eventualwithdraw./(1-eventualwithdraw)).*(1./( 1 + (growthrate./salesrate) ) ) ) ) ).*listingrate;

% Average rates (quarterly)
avmovingrate = mean(movingrate);
aveventualwithdraw = mean(eventualwithdraw);
avrelistever = mean(relistever);

%% Figure 4: Counterfactual transactions after accounting separately for
%% withdrawals

% Steady-state sales volume
cfwsales = (salesrate.*listingrate.*hstock)./(salesrate + listingrate + growthrate);

% Sales volume with variable sales rate
lstrvars = avmovingrate./(1 + (aveventualwithdraw/(1-aveventualwithdraw)).*(salesrate./(salesrate + avgrowthrate)).*( ...
    (avmovingrate + avgrowthrate)./( (avmovingrate./(1-avrelistever)) + avgrowthrate ) ) );
cfwsalesvars = (avhstock*salesrate.*lstrvars)./(salesrate + lstrvars + avgrowthrate);

% Sales volume with variable moving rate
lstrvarm = movingrate./(1 + (aveventualwithdraw/(1-aveventualwithdraw))*(avsalesrate./(avsalesrate + avgrowthrate)).*( ...
    (movingrate + avgrowthrate)./( (movingrate./(1-avrelistever)) + avgrowthrate ) ) );
cfwsalesvarm = (avsalesrate*avhstock*lstrvarm)./(avsalesrate + lstrvarm + avgrowthrate);

% Sales volume with variable withdrawals
lstrvarw = avmovingrate./(1 + (eventualwithdraw./(1-eventualwithdraw)).*(avsalesrate./(avsalesrate + avgrowthrate)).*( ...
    (avmovingrate + avgrowthrate)./( (avmovingrate./(1-relistever)) + avgrowthrate ) ) );
cfwsalesvarw = (avsalesrate*avhstock*lstrvarw)./(avsalesrate + lstrvarw + avgrowthrate);

% Sales volume with variable housing stock
cfwsalesvarg = (avsalesrate*avlistingrate.*hstock)./(avsalesrate + avlistingrate + growthrate);

% Log differences relative to initial value of each series
fig4 = 100*[(log(cfwsales) - log(cfwsales(1))) (log(cfwsalesvars) - log(cfwsalesvars(1))) (log(cfwsalesvarm) - log(cfwsalesvarm(1))) ... 
    (log(cfwsalesvarw) - log(cfwsalesvarw(1))) (log(cfwsalesvarg) - log(cfwsalesvarg(1)))];

% Plot figure
varlab = {'Vary all four';'Vary sales rate';'Vary moving rate';'Vary withdrawals';'Vary stock'}';
figure;
plot(datelab,fig4);
datetick('x','yyyy','keeplimits');
legend(varlab);
