% Produce impulse response functions

% Load parameters
load('paramendog2.mat');

% Sizes of shocks
boom = 0.23;
searcheff = 0.19;
mortgage = -0.30;

% Time spent in each state
yp = [1;20];

% Parameters for initial steady states
pinit = cell(1,3);
pinit{1} = paramvec;
pinit{2} = paramvec;
pinit{3} = paramvec;

% Changes to parameters over time for each distinct case
pvec = cell(2,3);
pvec{1,1} = paramvec;
pvec{1,2} = paramvec;
pvec{1,3} = paramvec;
pvec{2,1} = pvec{1,1};
pvec{2,1}{2}(9) = (1+boom)*pvec{2,1}{2}(9);
pvec{2,2} = pvec{1,2};
pvec{2,2}{2}(2) = (1+searcheff)*pvec{2,2}{2}(2);
pvec{2,3} = pvec{1,3};
pvec{2,3}{2}(8) = (1+mortgage)*pvec{2,3}{2}(8);

% Absolute starting time
ta = -2;

% Number of time periods per year
frq = 52;

%% Numbers of cases, regimes, years

% Number of distinct cases
nc = size(pvec,2);

% Number of parameter regimes
np = size(pvec,1);

% Total number of years
ny = sum(yp);

% Dates to evaluate solution
tau = (ta:(1/frq):ta+ny)';
nt = size(tau,1);
tp = [1; 1 + frq*cumsum(yp)];

% Store results
impresp = cell(1,nc);

%% Main loop over each case
for j=1:nc
    
    % Initial state

    % Unpackage parameters
    idio = pinit{j}{1};
    theta = idio{1};
    a = idio{2};
    dl = idio{3};
    q = idio{4};
    param = pinit{j}{2};
    lambda = param(1);
    v = param(2);
    C = param(3);
    F = param(4);
    D = param(5);
    kappa = param(6);
    omega = param(7);
    r = param(8);
    xi = param(9);

    % Solve model at initial parameters
    results = solvemodel(pinit{j});

    % Unpackage results
    x = results(1);
    y = results(2);
    pi = results(3);
    s = results(4);
    n = results(5);
    u = results(6);
    S = results(7);
    P = results(8);

    % Coefficients of dynamic system
    A = -[(-diag(a) - v*(y^(-lambda))*(theta*ones(1,q))) (v*(x^(-lambda))*diag(theta.*dl./(1-dl))); (-(a.*(1-dl))*ones(1,q)) -diag(a.*(1-dl)) ];
    b = [v*(y^(-lambda))*theta; a.*(1-dl)];

    % Matrices of impulse responses
    imprvec = zeros(nt,2*q);
    imprs = zeros(nt-1,1);
    imprP = zeros(nt-1,1);
    imprN = zeros(nt-1,1);

    % Start from implied steady state
    imprvec(1,:) = (A\b)';

    % Calculate impulse response functions

    % Loop for each parameter regime
    for i=1:np

        % New values of parameters
        paramvec0 = pvec{i,j};
        idio0 = paramvec0{1};
        param0 = paramvec0{2};
        theta0 = idio0{1};
        a0 = idio0{2};
        dl0 = idio0{3};
        lambda0 = param0(1);
        v0 = param0(2);
        C0 = param0(3);
        F0 = param0(4);
        D0 = param0(5);
        kappa0 = param0(6);
        omega0 = param0(7);
        r0 = param0(8);
        xi0 = param0(9);

        % Solve model at current parameters
        results0 = solvemodel(paramvec0);
        x0 = results0(1);
        y0 = results0(2);
        pi0 = results0(3);
        s0 = results0(4);
        n0 = results0(5);
        u0 = results0(6);
        S0 = results0(7);
        P0 = results0(8);

        % Coefficients of dynamic system
        A0 = -[(-diag(a0) - v0*(y0^(-lambda0))*(theta0*ones(1,q))) (v0*(x0^(-lambda0))*diag(theta0.*dl0./(1-dl0))); (-(a0.*(1-dl0))*ones(1,q)) -diag(a0.*(1-dl0)) ];
        b0 = [v0*(y0^(-lambda0))*theta0; a0.*(1-dl0)];

        % Implied steady state
        vec0 = A0\b0;

        % Eigenvalue factorization of dynamic matrix
        [V E] = eig(A0);
        ev = diag(E);
        c = V\(imprvec(tp(i),:)' - vec0);
        imprvec(tp(i)+1:tp(i+1),:) = ones(tp(i+1) - tp(i),1)*(vec0)' + real((V*(diag(c)*exp(-ev*((tau(tp(i)+1:tp(i+1))') - tau(tp(i)) ))))');

        % Sales rate
        imprs(tp(i):tp(i+1)-1) = v0*(y0^(-lambda0));

        % Average transactions price
        imprP(tp(i):tp(i+1)-1) = kappa0*C0 - (D0/r0) + omega0*((1/r0) + ((y0^lambda0)/v0))*( xi0*x0 + F0 );

        % New listings
        imprN(tp(i):tp(i+1)-1) = imprvec(tp(i):tp(i+1)-1,1:q)*a0 - (v0*(x0^(-lambda0)))*(imprvec(tp(i):tp(i+1)-1,q+1:2*q)*(theta0.*dl0./(1-dl0)));

    end

    % Houses for sale
    impru = 1 - sum(imprvec(1:nt-1,1:q),2);

    % Transactions
    imprS = imprs.*impru;

    % Moving rate
    imprn = imprN./(1-impru);

    % Convert impulse responses to log differences
    imprs = 100*(log(imprs) - log(imprs(1)));
    imprn = 100*(log(imprn) - log(imprn(1)));
    imprS = 100*(log(imprS) - log(imprS(1)));
    imprN = 100*(log(imprN) - log(imprN(1)));
    impru = 100*(log(impru) - log(impru(1)));
    imprP = 100*(log(imprP) - log(imprP(1)));
    
    % Store results
    impresp{j} = [imprS imprN imprs imprn impru imprP];
    
end

% Time axis
years = tau(1:nt-1);

% Package results
imprespS = [impresp{1}(:,1) impresp{2}(:,1) impresp{3}(:,1)];
imprespN = [impresp{1}(:,2) impresp{2}(:,2) impresp{3}(:,2)];
impresps = [impresp{1}(:,3) impresp{2}(:,3) impresp{3}(:,3)];
imprespn = [impresp{1}(:,4) impresp{2}(:,4) impresp{3}(:,4)];
imprespu = [impresp{1}(:,5) impresp{2}(:,5) impresp{3}(:,5)];
imprespP = [impresp{1}(:,6) impresp{2}(:,6) impresp{3}(:,6)];
impresp = {imprespS imprespN impresps imprespn imprespu imprespP};
nv = 6;
vnames = {'Transactions';'Listings';'Sales rate';'Moving rate';'Houses for sale';'Prices'};
shocknames = {'Productivity';'Search';'Mortgages'};

%% Display results

% Create figure
figure;
for k=1:nv
    
    % Create axes
    subplot(3,2,k);
    
    % Plot impulse response functions
    plot(years,impresp{k});
    
    % Title
    title(vnames{k});
    
    % Add legend
    if (k==3)
        legend(shocknames);
    end
    
end
