% Calibrate the model and save the matching parameters

%% Calibration and parameter options

% Filename to save parameters under
filename = 'paramendog2.mat';

% Number of types
q = 2;

% Exogenous moving special case
exog = false; % set to true to impose exogenous moving model (delta_i = 0)

% Calibration targets, other than hazard function
Ts = 6.5/12;
Vs = 10;
c = 0.1;
f = 0.025;
d = 0.045;
kappa = 1/3;
omega = 1/2;
r = 0.057;

% Weighting method
weightsmodelbased = false; % Set to true to use model-based weights and update iteratively

% Include fixed effects (time and cohort) in hazard function estimation
hazfe = false;

% Maximum duration of observations to use in hazard function estimation
% (years)
maxdur = 80;

%% Grid to search on

% Minimum and maximum values and numbers of grid points along each dimension 
prmin = 0.0001;
prmax = 0.95;
prnum = 12;
amin = 0.001;
amax = 1;
anum = 12;
dlmin = 0.0001;
dlmax = 0.8*amax;
dlnum = 12;

% Number of initial points for Nelder-Mead search
ng = 250;

%% Get empirical hazard function data

% Read in data
hazdata = empiricalhazard(maxdur,hazfe);
dur = hazdata{1}; % durations
hazemp = hazdata{2}; % hazard rates
nobv = hazdata{3}; % number of observations at each duration
empweights = hazdata{4};

% Weighting method
if weightsmodelbased
    load('weights');
    weights = newweights;
    hazdata{4} = weights;
else
    weights = empweights;
end

%% Grid search

% Package targets
targets = [Ts;Vs;c;f;d;kappa;omega;r];

% Construct grids
prstep = (prmax-prmin)/(prnum-1);
prgrid = (prmin:prstep:prmax)';
astep = (amax-amin)/(anum-1);
agrid = (amin:astep:amax)';
dlstep = (dlmax-dlmin)/(dlnum-1);
dlgrid = (dlmin:dlstep:dlmax)'; 

% Special case of exogenous moving
if exog
    dlexog = dlmin;
    dl0 = dlexog*ones(q,1);
else
    dlexog = NaN;
end

% Size of grid
if exog
    g = (prnum^(q-1))*(anum^q);
    m = 2*q - 1;
    f = [prnum*ones(q-1,1);anum*ones(q,1)];
else
    g = (prnum^(q-1))*(anum^q)*(dlnum^q);
    m = 3*q - 1;
    f = [prnum*ones(q-1,1);anum*ones(q,1);dlnum*ones(q,1)];
end
dist = zeros(g,1);

% Main loop
for i=1:g
    
    % Report progress
    if (rem(i,1000) == 0)
        disp(sprintf('Percentage complete: %g',100*i/g));
    end
    
    % Loop for index numbers
    idx = zeros(m,1);
    k = i-1;
    for j=m:-1:1
        idx(j) = rem(k,f(j)) + 1;
        k = fix(k/f(j));
    end
    
    % Parameter vectors
    pr0 = prgrid(idx(1:q-1));
    a0 = agrid(idx(q:2*q-1));
    if ~exog
        dl0 = dlgrid(idx(2*q:3*q-1));
        hazparam0 = [pr0;a0;dl0];
    else
        hazparam0 = [pr0;a0];
    end
    
    % Distance between model and empirical hazard functions
    dist(i) = hazdist(hazparam0,hazdata,targets,dlexog);
    
end

%% Nelder-Mead search

% Set up vectors
mindist = zeros(1,ng);
minhazparam = zeros(m,ng);

% Main loop
for h=1:ng
    
    % Find next smallest distance
    [mindist(h) i] = min(dist);
    disp(sprintf('Search: %g of %g starting from %g',h,ng,mindist(h)));
    dist(i) = NaN;
    
    % Loop for index numbers
    idx = zeros(m,1);
    k = i-1;
    for j=m:-1:1
        idx(j) = rem(k,f(j)) + 1;
        k = fix(k/f(j));
    end
    
    % Parameter vectors
    pr0 = prgrid(idx(1:q-1));
    a0 = agrid(idx(q:2*q-1));
    if ~exog
        dl0 = dlgrid(idx(2*q:3*q-1));
        hazparam0 = [pr0;a0;dl0];
    else
        hazparam0 = [pr0;a0];
    end
    
    % Nelder-Mead search
    [minhazparam(:,h) mindist(h)] = fminsearch(@hazdist,hazparam0,[],hazdata,targets,dlexog);
    
end

% Find minimum
[mindist2 hmin] = min(mindist);
hazparam = minhazparam(:,hmin);

%% Assemble estimated parameters and save the results

% Extract shock distribution parameter vectors
pr = hazparam(1:q-1);
a = hazparam(q:2*q-1);
if exog
    dl = dl0;
else
    dl = hazparam(2*q:3*q-1);
end
theta = cumprod([1;pr]);
theta = (1/sum(theta))*theta;

% Package shock distribution parameters
idio = {theta a dl q};

% Find parameters matching calibration targets
param = paramcalib(targets,idio);

% Parameter vector
paramvec = {idio; param};

% Save calibrated parameters
save(filename,'paramvec');

%% Solve the model and update weights

% Solve model
results = solvemodel(paramvec);

% Update weights by computing survival function
newweights = survival(paramvec,results,dur);
save('weights.mat','newweights');
