% Generate individual price paths in DSGE model with sales

%% Options

% Parameters of model of sales
epsilon = 3.01; % elasticity of substitution between product types
eta = 19.8; % bargain hunters' elasticity of substitution between brands
lambda = 0.735; % fraction of loyal customers for each brand
sigma = 0.255; % size of sector of economy with sales

% Other parameters (monthly calibration)
beta = 0.9975; % subjective discount factor
thetac = 0.333; % intertemporal elasticity of substitution in consumption
thetah = 0.7; % Frisch elasticity of labour supply
alpha = 0.667; % elasticity of output w.r.t. hours
gamma = (1-alpha)/alpha; % elasticity of marginal cost w.r.t. output
varsigma = 20; % elasticity of substitution between differentiated labour inputs
phip = 0.889; % probability of stickiness of normal prices
phiw = 0.889; % probability of wage stickiness
p = 0.536; % first-order serial correlation of money-supply growth rate

% Probability distribution of exogenous shocks
mps_sd = 0.02; % standard deviation of shock to money growth

% Number of price observations to generate
nt = 24; % length of time series for price paths
pobs = 4; % number of price observations per period
np = 10; % number of price paths to generate

%% Main code

% Indices of endogenous variables
ipibar = 1; % aggregate inflation
iYbar = ipibar + 1; % aggregate output
ii = iYbar + 1; % nominal interest rate
iw = ii + 1; % real wage
iwstar = iw + 1; % desired real wage
ipiW = iwstar + 1; % wage inflation
irho = ipiW + 1; % non-sale-sector price level to average normal price
ix = irho + 1; % real marginal cost in sale sector (defined using aggregate price level)
iY = ix + 1; % value-added of sale sector
iQ = iY + 1; % physical output of sale sector
iMg = iQ + 1; % growth rate of money supply

% Find steady state
[mu chi s sbar] = solvess(epsilon,eta,lambda,sigma);

% Obtain characteristics of equilibrium and coefficients
z = zroot(mu,epsilon,eta);
[psi x Delta delta] = equilaux(epsilon,eta,lambda,mu,s);
psibar = sigma*psi;

% Solve dynamic model
[Phi Theta] = solvedyn(epsilon,eta,lambda,sigma,beta,thetac,thetah,alpha,gamma,varsigma,phip,phiw,p);

% Calculate time series of aggregate variables for one realization of a
% sequence of exogenous shocks
shocks = mps_sd*randn(nt+1,1);
ns = size(Phi,1);
timeseries = zeros(nt,ns);
timeseries(1,:) = shocks(1)*Theta';
for t=2:nt
    timeseries(t,:) = timeseries(t-1,:)*Phi' + shocks(t)*Theta';
end

% Calculate time series for characteristics of price distribution
Pt = cumsum(timeseries(:,ipibar));
st = -((eta-1)/(mu^(1-eta) - 1))*((1-s)/(1-psibar))*(timeseries(:,ix) + (1-sigma)*timeseries(:,irho));
PSt = Pt + timeseries(:,ix);
PNt = Pt - (psibar/(1-psibar))*timeseries(:,ix) - ((1-sigma)/(1-psibar))*timeseries(:,irho);
RNt = (1/(1-phip))*(PNt - phip*[0; PNt(1:nt-1)]);

% Calculate indvidual price paths
nobs = pobs*nt;
pp = zeros(nobs,np);
for k=1:np
    pn = 0;
    for t=1:nt
        if (rand > phip)
            pn = RNt(t);
        end
        ps = PSt(t);
        sf = s + st(t) + ((1-s)*(epsilon + z*eta)/((mu^(-epsilon) - 1) + z*(mu^(-eta) - 1)))*(pn - PNt(t));
        for ob=1:pobs
            if (rand > sf)
                pp((t-1)*pobs+ob,k) = exp(pn);
            else
                pp((t-1)*pobs+ob,k) = mu*exp(ps);
            end
        end
    end
end
