// *********************************************************************************************************************
// * Table 1: Market segment level results

// * This file replicates results in Table 1 of the paper.
// * Run robustness checks for different periods and distances when specified in global variables.

//  First set global paths running 00_init.do

// Required global variables:
// - $path_data
// - $path_table_reg
// - $market_seg
// - $prop_type
// - $crisis_removed
// - $crisis_start
// - $crisis_end
// - $sample_number
// - $default_distances
// - $placebo_border
// - $yearmonthfe

// *********************************************************************************************************************

// *Notes: 

// global market_seg  defines not only if amrket level is used but with version of regressions are used. 
// 0 Regression in main table 1
// <0 Regression by property type
// >1 Regression by market segment, other controls

// - LTT is defined as post LTT period and in City of Toronto

// - toronto is defined as the City of Toronto, toronto = 1 if distance < 0

// - distance is defined as the distance from the neighborhood to the Toronto boundary in meters

// - post is defined as year month after or on March 2008

// - community (com) is  is zipcode or postal code

// - property types are: 5 categories
// Att/Row/Twnhouse |     17,361       13.47       13.47
//        Condo Apt |     27,172       21.08       34.55
//  Condo Townhouse |     21,424       16.62       51.17
//         Detached |     40,826       31.67       82.84
//    Semi-Detached |     22,123       17.16      100.00

// Sample definitions: 
// - sample3 is the main sample for the paper (Jan 2006 to Feb 2012)
// - sample4 is the sample for the robustness checks (Jan 2006 to Feb 2010)
// - sample6 is the sample for the extended analysis (Jan 2006 to Feb 2018)
// - sample8 is the sample for the extended analysis (Jan 2006 to Feb 2016)

// TO_3m is a categorical variable for year/month observations in City of Toronto +- 3 months around the event date (March 2008)

// - x_25 <gen x_25=LTT*(distance<-2500 & distance>-5000)>
// - x_25 is a dummy variable for the 2.5 km doughnut hole around the LTT boundary

// - LTT_distance is the distance only for LTT = 1 observations (post and in Toronto)


// *********************************************************************************************************************
// * Set up
if $dominant_log==0{
    cap log close  
    log using "$path_logs/03_table1.log", replace
}

// Community_18M_distance data

use "$path_data/Community_18M_distance.dta", clear

local filename = "table1"

local add = ""

local sample = "$sample_number"

local keep = "LTT"

local cluster_list = "com year PropertyType" //yearmonth year month  com year com PropertyType 

local cluster = "vce(robust)"

local main_indep_vars = "post toronto LTT i.TO_3m"

local market_segment = $market_seg // from -6 to 2 $market_seg

local property_type = $prop_type  // from 1 to 6 $prop_type

// * crisis
local crisis_remove = $crisis_removed // 1 to remove crisis period
local crisis_start = "$crisis_start" // "ym(2008,12)"
local crisis_end = "$crisis_end" //"ym(2009,12)"

// about distance thresholds and doughnut hole
local default_distances = $default_distances

local placebo_border = $placebo_border

local yearmonthfe = $yearmonthfe

local houseprice_tier = $houseprice_tier

// *********************************************************************************************************************
// * Robustness checks

// local add = "`add'_seg`market_segment'"

// * property types

local cond_ = ""
if `market_segment'==-1 | `market_segment'==-3 | `market_segment'==-6 {   

    // * property types
	local cond_ = "& PropertyType==`property_type'"

    // *  property types combinations 
    // * number 6 : Detached and Semi-Detached
	if `property_type'==6{
		local cond_ = "& (PropertyType==4| PropertyType==5)"
	}
    // * number 7 : Condo Apt and Condo Townhouse
    if `property_type'==7{
        local cond_ = "& (PropertyType==2| PropertyType==3)"
    }
    // * number 8 : Detached and Semi-Detached and Condo Apt and Condo Townhouse
    if `property_type'==8{
        local cond_ = "& (PropertyType==2| PropertyType==3| PropertyType==4| PropertyType==5)"
    }

    local add = "`add'_prop`property_type'"
}

if `crisis_remove'==1 {

    local add = "`add'_crisis_start`crisis_start'_end`crisis_end'"

    gen crisis_def= 1*(yearmonth>=`crisis_start' & yearmonth<=`crisis_end')

    drop if crisis_def==1
}


if `placebo_border'<0 {

    di "Placebo border: `placebo_border'"

    local add = "`add'_placebo`placebo_border'"

    sum distance 

    replace distance = distance - `placebo_border'

    gen placebo_tor_def = 1*(distance<0)
    gen LTT_placebo = LTT*(distance<0)
    sum distance placebo_tor_def LTT_placebo 

    drop if toronto==0

    replace toronto = placebo_tor_def
    replace LTT = LTT_placebo
    replace x_25 = (distance<-2500 & distance>-5000)

    drop placebo_tor_def LTT_placebo
}


// * default distances 3km, 5km, don 2km

if `default_distances'==1 {
    local min_dist = 3000 // typically 3 km
    local min_label = "3KM"

    local med_dist = 5000
    local med_dist_lab = "5KM"

    local doughnut_lab = "2KM"
    local doughnut_dist = 2000
}
if `default_distances'==2 {
    // for placebo test uses 2km, 4km, donut 1km
    local min_dist = 6000
    local min_label = "6KM"

    // 8
    local med_dist = 8000
    local med_dist_lab = "8KM"

    local doughnut_lab = "2KM"
    local doughnut_dist = 2000

    local add = "`add'_dist2"

}

if `default_distances'==4 {
    local min_dist = 4000 // typically 3 km
    local min_label = "4KM"

    local med_dist = 5000
    local med_dist_lab = "5KM"

    local doughnut_lab = "2KM"
    local doughnut_dist = 2000

    local add = "`add'_dist4"
}
if `default_distances'==0 {
    // for placebo test uses 2km, 4km, donut 1km
    local min_dist = 2000
    local min_label = "2KM"

    local med_dist = 4000
    local med_dist_lab = "4KM"

    local doughnut_lab = "1KM"
    local doughnut_dist = 1000
}
// * By median price

if `houseprice_tier'>0 {
    egen median_price_3km = median(Total_AvgSP) if abs(distance)<`min_dist' & `sample'==1 `cond', by(year)
    egen median_price_5k = median(Total_AvgSP) if abs(distance)<`med_dist' & `sample'==1 `cond', by(year)
    egen median_price = median(Total_AvgSP) if `sample'==1 `cond', by(year)
}

// * above price
if `houseprice_tier'==1 {
    local add = "`add'_hpabove"
    local cond_hp_3km = "& Total_AvgSP>median_price_3km"
    local cond_hp_5k = "& Total_AvgSP>median_price_5k"
    local cond_hp = "& Total_AvgSP>median_price"
}
// * below price
if `houseprice_tier'==2 {
    local add = "`add'_hpbelow"
    local cond_hp = "& Total_AvgSP<=median_price`dist'"
    local cond_hp_3km = "& Total_AvgSP<=median_price_3km"
    local cond_hp_5k = "& Total_AvgSP<=median_price_5k"
    local cond_hp = "& Total_AvgSP<=median_price"
}


// ******* Reg FE and controls *****************************************************************************************

local FE = "i.PropertyType##i.year i.toronto##i.PropertyType i.month##i.PropertyType i.com##i.PropertyType " //i.

// * additionally yearmonth -> move after each config outcome
if `yearmonthfe'==1 {
    local FE = "`FE' i.yearmonth" //i.  
    local add = "`add'_feym"
}

local control = "c.yearmonth#c.toronto"

local filename = "`filename'_`sample'`add'"

// *********************************************************************************************************************
// * Creating new variables

egen cluster_group = group(`cluster_list')

gen leasesales_ratio = TotalLease_LeaseCount/Total_SaleCount
gen ln_leasesales_ratio= log(leasesales_ratio)

// * winsorizing
foreach var in `winsor' {
    // qui winsor `var', gen(`var'_w) p(5 95)
	winsor2 ln_`var', cuts(0 95) by(year)
}


xtile pct_salecount = Total_SaleCount, nq(100)

// * BTO and BTR except for places with 98% percentile or more of the sales
gen ln_BTO_Count_E12=ln_BTO_SaleCount 
replace ln_BTO_Count_E12=. if pct_salecount>=98

gen ln_BTR_Count_E13=ln_BTR_SaleCount 
replace ln_BTR_Count_E13=. if pct_salecount>=97

xtile pct_BTRcount = BTR_SaleCount, nq(100)
gen ln_BTR_Count_E04=ln_BTR_SaleCount 
replace ln_BTR_Count_E04=. if pct_BTRcount>=95 


// *price to rent ratio with more than 1 sale and lease
gen ln_PRRatio_Avg=log(Total_PRRatio_Avg) 
gen ln_PRRatio_Avg_H2 = ln_PRRatio_Avg
replace ln_PRRatio_Avg_H2=. if Total_PRRatio_Avg==0 | TotalLease_LeaseCount==1 | Total_SaleCount==1

// *********************************************************************************************************************
// * Labels

local l_ln_BTO_SaleCount "log(BTR Sales)"
local l_ln_BTR_SaleCount "log(BTO Sales)"
local l_ln_BTO_SaleCount_w "log(BTR Sales)"
local l_ln_BTR_SaleCount_w "log(BTO Sales)"
local l_ln_Total_SaleCount "log(\# Sales)"
local l_ln_TotalLease_LeaseCount "log(\# Leases)"
local l_ln_Total_SaleCount_w "log(\# Sales)"
local l_ln_TotalLease_LeaseCount_w "log(\# Leases)"
local l_ln_BTO_Count_E12 "log(BTO Sales)"
local l_ln_BTR_Count_E13 "log(BTR Sales)"
local l_ln_BTR_Count_E04 "log(BTR Sales)"
local l_ln_leasesales_ratio "log(Lease/Sales)"
local l_ln_PRRatio_Avg "log(Price/Rent)"
local l_ln_PRRatio_Avg_H2 "log(Price/Rent)"
local l_ln_rent "log(Rent)"
local l_ln_rent_w "log(Rent)"
local l_ln_Total_AvgSP "log(Price)"


// *********************************************************************************************************************


di "Table 1: Replicating results from paper"

di "Filename: `filename'"

di "Sample: `sample'"
di "Sample number: `num'"
di "Market segment: `market_segment'"
di "Property type: `property_type'"
di "Crisis removed: `crisis_remove'"
di "Min distance: `min_dist'"
di "Cluster: `cluster'"
di "Cluster list: `cluster_list'"
di "Main independent variables: `main_indep_vars'"
di "Additional: `add'"
di "Keep: `keep'"
di "Control: `control'"
di "cond: `cond_'"
di "Default distances: `default_distances'"
di "Distance thresholds: `min_dist' `med_dist' `doughnut_dist'"
di "Placebo border: `placebo_border'"
di "Yearmonth FE: `yearmonthfe'"
di "House price tier: `houseprice_tier'"


// *********************************************************************************************************************
// * Regressions  Table 1
// * *********************************************************************************************************************

// Default main specification 
if `market_segment'==0 { 
    local cond = " & PropertyType==4"
    local FE = "i.year i.month i.com i.toronto i.com#i.year"
    local control = "c.yearmonth#c.toronto"
} 
else { 
    di "Property type: `property_type'"
    local cond = " `cond_'" // "`cond_'" `property_type'
}

// *********************************************************************************************************************
// * Lease/Sales Ratio
// *********************************************************************************************************************


if `market_segment'==-2 { 
    local cond = " & PropertyType==4"
} 
if `market_segment'==-3 {  // means property type specified, control changes
    local FE = "i.toronto i.month i.com i.PropertyType"
    local control = "c.yearmonth#c.toronto"
}
if `market_segment'==-6 {  // this is for the new property type 6
    local FE = " i.toronto#i.PropertyType i.month##i.PropertyType i.com##i.PropertyType i.year##i.PropertyType  i.yearmonth"
    local control = "c.yearmonth#c.toronto"
}
if `yearmonthfe'==1 {
    local FE = "`FE' i.yearmonth" //i.  
}

// local control = "c.yearmonth#c.toronto"

local depvar = "ln_leasesales_ratio" // from main result of paper table

if `houseprice_tier'>0 {

    local c2 = "& `depvar'!=."

    drop median_price_3km median_price_5k median_price
    egen median_price_3km = median(Total_AvgSP) if abs(distance)<`min_dist' & `sample'==1 `cond' `c2', by(year)
    egen median_price_5k = median(Total_AvgSP) if abs(distance)<`med_dist' & `sample'==1 `cond' `c2', by(year)
    egen median_price = median(Total_AvgSP) if `sample'==1 `cond' `c2', by(year)
}


eststo clear

if `crisis_remove'==0 {
*2
qui reghdfe `depvar' `main_indep_vars' `control' if `sample'==1 & abs(distance)<`min_dist' `cond' `cond_hp_3km', a(`FE') `cluster' //c!luster was commented!
eststo est2
estadd local row_dist "`min_label'":est2
estadd local row_TO_3m "YES":est2
estadd local row_time_trend "YES":est2
}

*5
qui reghdfe `depvar' `main_indep_vars' x_25 `control' if `sample'==1 & abs(distance)<`med_dist' `cond' `cond_hp_5km', a(`FE') `cluster' //!cluster was commented!
eststo est5
estadd local row_dist "`med_dist_lab'":est5
estadd local row_TO_3m "YES":est5
estadd local row_dist_trend "Step":est5
estadd local row_time_trend "YES":est5

*6
qui reghdfe `depvar' `main_indep_vars' x_25 `control' if `sample'==1 & abs(distance)<`med_dist' & abs(distance)>=2000 `cond' `cond_hp_5km', a(`FE') `cluster'
eststo est6
estadd local row_dist "`med_dist_lab'":est6
estadd local row_TO_3m "YES":est6
estadd local row_dist_trend "Step":est6
estadd local row_time_trend "YES":est6
estadd local row_donut "`doughnut_lab'":est6
if `crisis_remove'==0 {
    if `placebo_border'==0{
        *8
        qui reghdfe `depvar' `main_indep_vars' LTT_distance `control' if `sample'==1  `cond' `cond_hp', a(`FE') `cluster'
        eststo est8
        estadd local row_dist "ALL":est8
        estadd local row_TO_3m "YES":est8
        estadd local row_time_trend "YES":est8
        estadd local row_dist_trend "YES":est8
    }
    else {
        // assume 5km distance
        qui reghdfe `depvar' `main_indep_vars' `control' x_25 if `sample'==1 & abs(distance)<5000 `cond', a(`FE') `cluster' 
        eststo est8
        estadd local row_dist "5KM":est8
        estadd local row_TO_3m "YES":est8
        estadd local row_time_trend "YES":est8
        estadd local row_dist_trend "Step":est8

    }
}
label variable LTT "`l_`depvar''"

esttab using "$path_table_reg/`filename'.tex", nogaps se star(* 0.1 ** 0.05 *** 0.001) nonumbers nomtitles nonotes keep(`keep') drop(LTT_distance,relax) label replace


// *********************************************************************************************************************
// * Price/Rent Ratio
// *********************************************************************************************************************


local keep = "LTT"

if `market_segment'==-2 { 
    local cond = "" // When need to use all properties because small sample. 
} 
if `market_segment'==-3 {  // means property type specified, 
    local FE = "i.year##i.PropertyType i.toronto##i.PropertyType i.month##i.PropertyType i.com##i.PropertyType"
    local control = "" 
}
if `market_segment'==-6 {  // this is for the new property type 6
    local FE = "i.year##i.PropertyType i.toronto##i.PropertyType i.month##i.PropertyType i.com##i.PropertyType"
    local control = "" //c.yearmonth#c.toronto // When no city level controls
    local cond = ""
}
if `yearmonthfe'==1 {
    local FE = "`FE' yearmonth" 
}
if `crisis_remove'==0 {

local depvar = "ln_PRRatio_Avg_H2"

if `houseprice_tier'>0 {
    
    local c2 = "& `depvar'!=."

    drop median_price_3km median_price_5k median_price
    egen median_price_3km = median(Total_AvgSP) if abs(distance)<`min_dist' & `sample'==1 `cond' `c2', by(year)
    egen median_price_5k = median(Total_AvgSP) if abs(distance)<`med_dist' & `sample'==1 `cond' `c2', by(year)
    egen median_price = median(Total_AvgSP) if `sample'==1 `cond' `c2', by(year)
}


eststo clear

if `property_type'!=5 & (`property_type'!=1 | `yearmonthfe'==0 | `market_segment'!=0) & `crisis_remove'==0 {
    *2
    qui reghdfe `depvar' `main_indep_vars' `control' if `sample'==1 & abs(distance)<`min_dist' `cond' `cond_hp_3km' , a(`FE') `cluster'
    eststo est2
    estadd local row_dist "`min_label'":est2
    estadd local row_TO_3m "YES":est2
    estadd local row_time_trend "YES":est2

    *5 
    qui reghdfe `depvar' `main_indep_vars' x_25 `control' if `sample'==1 & abs(distance)<`med_dist' `cond' `cond_hp_5km', a(`FE') `cluster'
    eststo est5
    estadd local row_dist "`med_dist_lab'":est5
    estadd local row_TO_3m "YES":est5
    estadd local row_dist_trend "Step":est5
    estadd local row_time_trend "YES":est5


    *6
    qui reghdfe `depvar' `main_indep_vars' x_25 `control' if `sample'==1 & abs(distance)<`med_dist' & abs(distance)>=2000 `cond' `cond_hp_5km', a(`FE') `cluster'
    eststo est6
    estadd local row_dist "`med_dist_lab'":est6
    estadd local row_TO_3m "YES":est6
    estadd local row_dist_trend "Step":est6
    estadd local row_time_trend "YES":est6
    estadd local row_donut "`doughnut_lab'":est6

    if `placebo_border'==0{
    *8

        qui reghdfe `depvar' `main_indep_vars' LTT_distance `control' if `sample'==1  `cond' `cond_hp', a(`FE') `cluster'
        eststo est8
        estadd local row_dist "ALL":est8
        estadd local row_TO_3m "YES":est8
        estadd local row_time_trend "YES":est8
        estadd local row_dist_trend "YES":est8 // changed last row from dist tp row_dist_trend
    }
    else {
        // assume 5km distance
        qui reghdfe `depvar' `main_indep_vars' `control' x_25 if `sample'==1 & abs(distance)<5000 `cond', a(`FE') `cluster' 
        eststo est8
        estadd local row_dist "5KM":est8
        estadd local row_TO_3m "YES":est8
        estadd local row_time_trend "YES":est8
        estadd local row_dist_trend "Step":est8

    }
    label variable LTT "`l_`depvar''"

    esttab using "$path_table_reg/`filename'.tex", nogaps se star(* 0.1 ** 0.05 *** 0.001) nonumbers nomtitles nonotes keep(`keep') drop(LTT_distance,relax) label append
}
}



// *********************************************************************************************************************
// * BTO
// *********************************************************************************************************************


if `market_segment'==-2 { 
    local cond = " & PropertyType==4"
} 
if `market_segment'==-3 {  
    local FE = "i.month i.com i.toronto i.PropertyType"
    local control = "c.yearmonth#c.toronto" // new for prop 6

}
if `market_segment'==-6 {  // this is for the new property type 6

    local FE = " i.month##i.PropertyType i.com##i.PropertyType i.toronto##i.PropertyType" // new for prop 6 
    local control = "c.yearmonth#c.toronto"
}
if `yearmonthfe'==1 {
    local FE = "`FE' i.yearmonth" //i.  
}

local depvar = "ln_BTO_Count_E12" // from main result of paper table

if `houseprice_tier'>0 {
    
    local c2 = "& `depvar'!=."

    drop median_price_3km median_price_5k median_price
    egen median_price_3km = median(Total_AvgSP) if abs(distance)<`min_dist' & `sample'==1 `cond' `c2', by(year)
    egen median_price_5k = median(Total_AvgSP) if abs(distance)<`med_dist' & `sample'==1 `cond' `c2', by(year)
    egen median_price = median(Total_AvgSP) if `sample'==1 `cond' `c2', by(year)
}

eststo clear

if `crisis_remove'==0 {
*2
qui reghdfe `depvar' `main_indep_vars' `control' if `sample'==1 & abs(distance)<`min_dist' `cond' `cond_hp_3km', a(`FE') `cluster'
eststo est2
estadd local row_dist "`min_label'":est2
estadd local row_TO_3m "YES":est2
estadd local row_time_trend "YES":est2
}
*5
qui reghdfe `depvar' `main_indep_vars' x_25 `control' if `sample'==1 & abs(distance)<`med_dist' `cond' `cond_hp_5km', a(`FE') `cluster'
eststo est5
estadd local row_dist "`med_dist_lab'":est5
estadd local row_TO_3m "YES":est5
estadd local row_dist_trend "Step":est5
estadd local row_time_trend "YES":est5


*6 
qui reghdfe `depvar' `main_indep_vars' x_25 `control' if `sample'==1 & abs(distance)<`med_dist' & abs(distance)>=`doughnut_dist' `cond' `cond_hp_5km' , a(`FE') `cluster'
eststo est6
estadd local row_dist "`med_dist_lab'":est6
estadd local row_TO_3m "YES":est6
estadd local row_dist_trend "Step":est6
estadd local row_time_trend "YES":est6
estadd local row_donut "`doughnut_lab'":est6

*8
if `crisis_remove'==0 {
    if `placebo_border'==0{
        qui reghdfe `depvar' `main_indep_vars' LTT_distance `control' if `sample'==1  `cond' `cond_hp', a(`FE') `cluster'
        eststo est8
        estadd local row_dist "ALL":est8
        estadd local row_TO_3m "YES":est8
        estadd local row_time_trend "YES":est8
        estadd local row_dist_trend "YES":est8
    }
    else {
        // assume 5km distance 
        qui reghdfe `depvar' `main_indep_vars' `control' x_25 if `sample'==1 & abs(distance)<5000 `cond', a(`FE') `cluster' 
        eststo est8
        estadd local row_dist "5KM":est8
        estadd local row_TO_3m "YES":est8
        estadd local row_time_trend "YES":est8
        estadd local row_dist_trend "Step":est8

    }
}
label variable LTT "`l_`depvar''"

esttab using "$path_table_reg/`filename'.tex", nogaps se star(* 0.1 ** 0.05 *** 0.001) nonumbers nomtitles nonotes keep(`keep') drop(LTT_distance,relax) label append

// *********************************************************************************************************************
// * BTR
// *********************************************************************************************************************

local depvar = "ln_BTR_Count_E13" // from main result of paper table

if `market_segment'==-2 { 
    local cond = " & PropertyType==4"
    local control = ""
    // because BTR is not well populated, we test eliminating the time trends
} 
if `market_segment'==-3 {  // means property type specified, controls like in paper
    // local FE = "i.year i.month i.com i.toronto i.PropertyType"
    local cond = " `cond_'" // "`cond_'"
    local FE = "i.year i.month i.com i.toronto" //* paper version if only SFH
    local control = "" //"c.yearmonth#c.toronto"
}
if `market_segment'==-6 {  // this is for the new property type 6
    local cond = " `cond_'" // "`cond_'"
    local FE = "i.toronto##i.PropertyType i.month##i.PropertyType i.com##i.PropertyType i.year##i.PropertyType "
    local control = "" //"c.yearmonth#c.toronto"
}
if `houseprice_tier'>0 {
    drop median_price_3km median_price_5k median_price
    egen median_price_3km = median(Total_AvgSP) if abs(distance)<`min_dist' & `sample'==1 `cond', by(year)
    egen median_price_5k = median(Total_AvgSP) if abs(distance)<`med_dist' & `sample'==1 `cond', by(year)
    egen median_price = median(Total_AvgSP) if `sample'==1 `cond', by(year)
}

if `yearmonthfe'==1 {
    local FE = "`FE' i.yearmonth" //i.  
}

eststo clear

// * est 2

if `houseprice_tier'>0 {
    
    local c2 = "& `depvar'!=."

    drop median_price_3km median_price_5k median_price
    egen median_price_3km = median(Total_AvgSP) if abs(distance)<`min_dist' & `sample'==1 `cond' `c2', by(year)
    egen median_price_5k = median(Total_AvgSP) if abs(distance)<`med_dist' & `sample'==1 `cond' `c2', by(year)
    egen median_price = median(Total_AvgSP) if `sample'==1 `cond' `c2', by(year)
}


if (`property_type'!=1 | `yearmonthfe'==0 | `market_segment'!=0) {

if `crisis_remove'==0 {
qui reghdfe `depvar' `main_indep_vars' `control' if `sample'==1 & abs(distance)<`min_dist' `cond' `cond_hp_3km', a(`FE') `cluster'
eststo est2
estadd local row_dist "`min_label'":est2
estadd local row_TO_3m "YES":est2
estadd local row_time_trend "YES":est2
}
// * est 5

qui reghdfe `depvar' `main_indep_vars' x_25 `control' if `sample'==1 & abs(distance)<`med_dist' `cond' `cond_hp_5km', a(`FE') `cluster'
eststo est5
estadd local row_dist "`med_dist_lab'":est5
estadd local row_TO_3m "YES":est5
estadd local row_dist_trend "Step":est5
estadd local row_time_trend "YES":est5

// * est 6

qui reghdfe `depvar' `main_indep_vars' x_25 `control' if `sample'==1 & abs(distance)<`med_dist' & abs(distance)>=`doughnut_dist' `cond_hp_5km' `cond', a(`FE') `cluster'
eststo est6
estadd local row_dist "`med_dist_lab'":est6
estadd local row_TO_3m "YES":est6
estadd local row_dist_trend "Step":est6
estadd local row_time_trend "YES":est6
estadd local row_donut "`doughnut_lab'":est6

// * est 8
if `crisis_remove'==0 {
    if `placebo_border'==0{

        // from main result of paper table
        qui reghdfe `depvar' `main_indep_vars' LTT_d25 LTT_d50 LTT_d75  `control' if `sample'==1  `cond' `cond_hp', a(`FE') `cluster' //! different distance controls
        eststo est8
        estadd local row_dist "ALL":est8
        estadd local row_TO_3m "YES":est8
        estadd local row_time_trend "YES":est8
        estadd local row_dist_trend "YES":est8
    }
    else {
        // assume 5km distance
        qui reghdfe `depvar' `main_indep_vars' `control' x_25 if `sample'==1 & abs(distance)<5000 `cond', a(`FE') `cluster' 
        eststo est8
        estadd local row_dist "5KM":est8
        estadd local row_TO_3m "YES":est8
        estadd local row_time_trend "YES":est8
        estadd local row_dist_trend "Step":est8

    }
}

label variable LTT "`l_`depvar''"

esttab using "$path_table_reg/`filename'.tex", nogaps se star(* 0.1 ** 0.05 *** 0.001) nonumbers nomtitles nonotes keep(`keep')  label append ///
			scalars("row_dist Distance threshold" "row_TO_3m Indicators TO +-3 m." "row_time_trend City time trend" "row_dist_trend Distance LTT trends"  "row_donut Donut Hole" ) 	
}

// *********************************************************************************************************************

if $dominant_log==0{
    log close
}