


// *********************************************************************************************************************
// This file: 08_codes_final/04_table2_hazard.do, produces: 
// * Table 2: Hazard regressions 
// * Hazard regressions robustness checks

// * This file replicates results in Table 2 of the paper.

//  First: set global paths running init.do
//  Second: run global set up variables in 01_run.do

// Required global variables:
// - $path_data
// - $path_table_reg
// - $sample_number
// - $market_seg
// - $prop_type
// - $crisis_removed
// - $default_distances
// - $placebo_border
// - $yearmonthfe

// Required files:
// - survival_data_final.dta

//*********************************************************************************************************************
// * Notes:
// Sample definitions: 
// - sample2 is the sample for the main results (Jan 2007 to Feb 2009)
// - sample3 is the main sample for the paper (Jan 2006 to Feb 2012)
// - sample4 is the sample for the robustness checks (Jan 2006 to Feb 2010)
// - sample6 is the sample for the extended analysis (Jan 2006 to Feb 2018)

// Property Types in data:
// differ by data set, check data set for property types below. 

// Property Types in Survival data:
// SEMI DETACHED | SINGLE FAMILY HOUSE | TOWNHOUSE

// controls code:
// seg 1: All properties, stronger controls
// seg 0: Paper version, main table
// *********************************************************************************************************************
if $dominant_log==0{
    cap log close  
    log using "$path_logs/04_table2_hazard.log", replace
}

// * Set up

local filename_base = "table2_"
// filename 

local add = ""

local sample = "$sample_number" // global : sample_number // default "sample3"

local keep = "LTT"

// local cluster_list = "yearmonth com"

// local cluster = "vce(robust)" //vce(robust)  cluster(cluster_group)

local main_indep_vars = "post toronto LTT i.TO_3m"

local market_segment = $market_seg // 1 if market segment on (main reg in paper), 0 if market segment off by property type
// local market_segment = 0 

local property_type = $prop_type  // from 1 to 6 $prop_type

// * crisis
local crisis_remove = $crisis_removed // 1 to remove crisis period

local count_rows = 0 // count rows whoch outcomes in table

// about distance thresholds and doughnut hole
local default_distances = $default_distances

local placebo_border = $placebo_border

local max_months_drop = 18 // default is 18

// * crisis

local crisis_start = "$crisis_start" // "ym(2008,12)"
local crisis_end = "$crisis_end" //"ym(2009,12)"

local yearmonthfe = $yearmonthfe

local hazard_off = $hazard_off // 1 to turn off hazard regressions

// ********************************************************************************************************************

di "Running table 2 replication"

di "Filename Base: `filename_base'"
di "Sample: `sample'"
di "Property Type: `property_type'"
di "Crisis: `crisis_remove'"
di "Keep: `keep'"
di "Default Distances: `default_distances'"
di "Placebo Border: `placebo_border'"
di "Year Month FE: `yearmonthfe'"
di "Max Months Drop: `max_months_drop'"
di "Cluster: `cluster'"

// ********************************************************************************************************************
// * Moving hazard regressions. Outcome: Event of moving 
// ********************************************************************************************************************

local filename = "`filename_base'" // commented

if `hazard_off'==1 {
    // # keep only representative sample 
    use in 1/10000 using "$path_data/survival_data_final.dta", clear
    local add = "`add'_hazard_off"
}
else {
    use "$path_data/survival_data_final.dta", clear
}

// *********************************************************************************************************************
// * Survival data Cleaning

** SAMPLES **
gen sample2=1*(yearmonth0>=ym(2007,01) & yearmonth0<=ym(2009,02))
gen sample3=1*(yearmonth0>=ym(2006,01) & yearmonth0<=ym(2012,02))
gen sample4=1*(yearmonth0>=ym(2006,01) & yearmonth0<=ym(2010,02))
gen sample6=1*(yearmonth0>=ym(2006,01) & yearmonth0<=ym(2018,02))

gen log_val = log(initial_val)

stset yearmonth1, id(homeid) time0(yearmonth0) origin(yearmonth0) failure(event==1)

* Drop flippers
sort homeid yearmonth1
by homeid: egen max_months = max(_t)
drop if max_months <= `max_months_drop'

egen d_max=max(abs(distance))

gen d_25=1*(abs(distance)/d_max <= 0.25)
gen d_50=1*(abs(distance)/d_max>0.25 & abs(distance)/d_max <=0.5)
gen d_75=1*(abs(distance)/d_max>0.5 & abs(distance)/d_max <=0.75)

gen LTT_d25=LTT*d_25
gen LTT_d50=LTT*d_50
gen LTT_d75=LTT*d_75

encode TYPE, gen(PropertyType)

gen com_=substr(zipcode,1,3)
egen com = group(com_)

// *********************************************************************************************************************
// * Robustness checks

// * crisis

if `crisis_remove'==1 {

    local add = "`add'_crisis_start`crisis_start'_end`crisis_end'"

	gen crisis_def = 0
	replace crisis_def = 1 if yearmonth0>=`crisis_start' & yearmonth0<=`crisis_end'

    drop if crisis_def==1

    local max_months_drop = 12 // shorter period to compensate for missing obs in crisis period
}

// * property types

local cond_ = ""
if `market_segment'<=-1  { // property specific

    // keep if PropertyType==`property_type'
	if  8==`property_type'{  //SEMI DETACHED
		local cond_ = "& PropertyType==1"
	} 
	if  9==`property_type'{ //SFH
		local cond_ = "& PropertyType==2"
	}
	if  10==`property_type'{ //TOWNHOUSE
	local cond_ = "& PropertyType==3"
	}
    if  11==`property_type'{ // DETACHED + SEMI DETACHED
        local cond_ = "& (PropertyType==1 | PropertyType==2)"
    }

    local add = "`add'_prop`property_type'"
}
local cond = "`cond' `cond_'"

if `market_segment'==0 | `market_segment'==1 {
    keep if TYPE=="SINGLE FAMILY HOUSE" 
}

if `placebo_border'<0 {

    di "Placebo border: `placebo_border'"

    local add = "`add'_placebo`placebo_border'"

    sum distance 

    replace distance = distance - `placebo_border'

    gen placebo_tor_def = 1*(distance<0)
    gen LTT_placebo = LTT*(distance<0)
    sum distance placebo_tor_def LTT_placebo 

    drop if toronto==0

    replace toronto = placebo_tor_def
    replace LTT = LTT_placebo
    replace x_25 = (distance<-2500 & distance>-5000)

    drop placebo_tor_def LTT_placebo
	
	// * new 
	local main_indep_vars = "`main_indep_vars' distance"
}

if `yearmonthfe'==1 {
    local FE = "`FE' i.yearmonth" 
    local add = "`add'_feym"
}

local var_price = "initial_val"

// default distnces 3km, 5km, don 2km

if `default_distances'==1 {
    local min_dist = 3000 // typically 3 km
    local min_label = "3KM"

    local med_dist = 5000
    local med_label = "5KM"

    local doughnut_lab = "2KM"
    local doughnut_dist = 2000

}
if `default_distances'==2 {
    // for placebo test uses 2km, 4km, donut 1km
    local min_dist = 6000
    local min_label = "6KM"

    local med_dist = 8000
    local med_label = "8KM"

    local doughnut_lab = "2KM"
    local doughnut_dist = 2000

    local add = "`add'_dist2"

}
if `default_distances'==0 {
    // for placebo test uses 2km, 4km, donut 1km
    local min_dist = 2000
    local min_label = "2KM"

    local med_dist = 4000
    local med_label = "4KM"

    local doughnut_lab = "1KM"
    local doughnut_dist = 1000

    local add = "`add'_dist0"
}
local filename = "`filename'_`sample'`add'"


// *****************************************************************************************************************
local control = "X_heating X_basement X_family X_fire X_beds X_baths X_kitch X_rooms X_lot c.yearmonth0 c.toronto#c.yearmonth0"
local FE = "i.PropertyType i.com"
local control = "`control' `FE'"

// ********************************************************************************************************************
// * Mobility (hazard)
// ********************************************************************************************************************
// *********************************************************************************************************************
//  * Regressions
if abs(`market_segment')==1 {
    local control = "" 
}

if abs(`market_segment')==2 { 
  local control = "X_heating X_basement X_family X_fire X_beds X_baths X_kitch X_rooms X_lot c.toronto#c.yearmonth0"
	local FE = "i.PropertyType i.com"
	local control = "`control' `FE'"
}

if abs(`market_segment') ==11{ 
  local control = "X_heating X_basement X_family X_fire X_beds X_baths X_kitch X_rooms X_lot yearmonth0 c.toronto#c.yearmonth0"
	local FE = "i.PropertyType i.com"
		local control = "`control' `FE'"
}

if abs(`market_segment') ==12{ 
  local control = "X_heating X_basement X_family X_fire X_beds X_baths X_kitch X_rooms X_lot toronto##c.yearmonth0"
	local FE = "i.PropertyType i.com" 
	local control = "`control' `FE'"
}

eststo clear

local  depvar = "log_val"

// label  depvar = "log (Original purchase price)"

label variable LTT  "LTT (Event of moving)"

if `hazard_off'==0{

    *3
    if `crisis_remove'==0 {
    qui streg `depvar' `main_indep_vars' `control' if `sample'==1 & abs(distance)<`min_dist' `cond' `cond_hp_3km' , dist(w) vce(cluster zipcode) nohr
    eststo est3
    estadd local row_dist "`min_label'":est3
    estadd local row_TO_3m "YES":est3
    estadd local row_time_trend "YES":est3
    }
    *5
    qui streg `depvar' `main_indep_vars' x_25 `control' if `sample'==1 & abs(distance)<`med_dist' `cond' `cond_hp_5km', dist(w) vce(cluster zipcode) nohr
    eststo est5
    estadd local row_dist "`med_label'":est5
    estadd local row_TO_3m "YES":est5
    estadd local row_dist_trend "Step":est5
    estadd local row_time_trend "YES":est5


    *6
    qui streg `depvar' `main_indep_vars' x_25 `control' if `sample'==1 & abs(distance)<`med_dist' & abs(distance)>=`doughnut_dist' `cond' `cond_hp_5km', dist(w) vce(cluster zipcode) nohr
    eststo est6
    estadd local row_dist "`med_label'":est6
    estadd local row_TO_3m "YES":est6
    estadd local row_dist_trend "Step":est6
    estadd local row_time_trend "YES":est6
    estadd local row_donut "`doughnut_lab'":est6

    *8
    if `crisis_remove'==0 {
    if `placebo_border'==0 {
        qui streg `depvar' `main_indep_vars' LTT_distance `control' if `sample'==1 `cond' `cond_hp', dist(w) vce(cluster zipcode) nohr
        eststo est8
        estadd local row_dist "ALL":est8
        estadd local row_TO_3m "YES":est8
        estadd local row_time_trend "YES":est8
        estadd local row_dist_trend "Step":est8
    }
    else {
        // assume 5km distance
        qui streg `depvar' `main_indep_vars' `control' if `sample'==1 & abs(distance)<5000, dist(w) vce(cluster zipcode) nohr
        eststo est8
        estadd local row_dist "5KM":est8
        estadd local row_TO_3m "YES":est8
        estadd local row_time_trend "YES":est8
        estadd local row_dist_trend "Step":est8
    }
    }
    
    esttab using "$path_table_reg/`filename'.tex", nogaps se star(* 0.1 ** 0.05 *** 0.001) nonumbers nomtitles nonotes drop(LTT_distance post toronto *.TO_3m x_25, relax) label replace ///
                scalars("row_dist Distance threshold" "row_TO_3m Indicators TO +-3 m." "row_time_trend City time trend" "row_dist_trend Distance LTT trends"  "row_donut Donut Hole" )
    

    local count_rows = `count_rows' + 1
}

// ********************************************************************************************************************

if $dominant_log==0{
    log close
}