function prm = calibration(tgt)
% Calibration of model parameters

disp('Performing calibration...');

%% Parameters implied directly by the targets

% Population inflow
prm.chi = tgt.chi;

% Costs
prm.D = tgt.D_PP*tgt.PP;
prm.Co = tgt.Co_PP*tgt.PP;
prm.Ch = tgt.Ch_PPh*tgt.PP/( 1 - tgt.ii + tgt.ii*tgt.PPi_PPh );
prm.Ci = tgt.Ci_PPi*tgt.PPi_PPh*tgt.PP/( 1 - tgt.ii + tgt.ii*tgt.PPi_PPh );
prm.Dl = tgt.Dl_RR*tgt.PPi_PPh*tgt.PP/(tgt.PPi_RR*(1 - tgt.ii + tgt.ii*tgt.PPi_PPh));
prm.Cl = tgt.Cl_RR*tgt.PPi_PPh*tgt.PP/(tgt.PPi_RR*(1 - tgt.ii + tgt.ii*tgt.PPi_PPh));

%% Ratios and quantities implied by the targets

% Fraction of investors among buyers and measures of groups
ps = tgt.ii/(tgt.ii + (1-tgt.ii)*tgt.LLambdah);
qh = tgt.n*tgt.h*tgt.Tmh/( tgt.Tmh + tgt.Tbh );
uo = tgt.n*tgt.h*tgt.Tso/( (1-tgt.ii)*(tgt.Tmh + tgt.Tbh) );
ql = (1 - qh - uo)*tgt.Tml/(tgt.Tml + tgt.Tsl);
if (ql <= 0)
    disp('Calibration failed: Cannot get a well-defined measure of renters');
    return;
end
ul = (1 - qh - uo)*tgt.Tsl/(tgt.Tml + tgt.Tsl);
thetao = (tgt.Tbh/tgt.Tso)/(1 - ps + ps*tgt.LLambdah);
thetal = (tgt.n - qh - ql - (1-ps)*thetao*uo)/ul;
if (thetal <= 0)
    disp('Calibration failed: Cannot get a well-defined time-to-rent value');
    return;
end
Tbl = thetal*tgt.Tsl;

%% Exit rates, demographics, and transitions to homeownership

% Investors' exit rate
prm.rhol = ((tgt.ii/(1-tgt.ii))/((1-tgt.n*tgt.h)/(tgt.n*tgt.h)))*(1/tgt.Tmh)/( 1 + (tgt.Tbh - tgt.n*tgt.h*(tgt.Tso/(1-tgt.ii)))/((1-tgt.n*tgt.h)*tgt.Tmh) );

% Households' exit rate
prm.rho = tgt.phi/(tgt.Tmh + (1-tgt.phi)*tgt.Tbh);

% Moving rates within the city
mh = ((1-tgt.phi)*(tgt.Tmh + tgt.Tbh))/(tgt.Tmh*(tgt.Tmh + (1-tgt.phi)*tgt.Tbh));
ml = (1/tgt.Tml) - prm.rho;
if (ml <= 0)
    disp('Calibration failed: Cannot match time-to-move in rental market');
    return;
end

% Arrival rate of shocks for renters
prm.alphal = ml - prm.rhol;
if (prm.alphal <= 0)
    disp('Calibration failed: Landlord exit rate inconsistent with renter time-to-move');
    return;
end

% Probability of drawing new credit cost for renters who move
prm.xi = (prm.rho^2)*tgt.aleph*((1 + (tgt.Tml/Tbl))^2)/((tgt.Tml/Tbl)*((1/tgt.Tml) - prm.rho)*( prm.rho*tgt.Tml + (1 - prm.rho*tgt.aleph)*(1 + (tgt.Tml/Tbl)) ) );
if (prm.xi >= 1)
    disp('Calibration failed: Cannot match average ages of owner-occupiers and renters');
    return;
end

% Probability that credit cost is low enough to become home-buyer
kappa = prm.rho*tgt.n*tgt.h/(prm.xi*ml*ql + prm.rho*tgt.n);

%% Distribution of credit costs

% Marginal and average credit costs
Z = (tgt.PP/(1 - tgt.ii + tgt.ii*tgt.PPi_PPh))*((tgt.rz - tgt.rg)*tgt.l/((tgt.rg+prm.rho)*(1-exp(-tgt.rz*tgt.Tk))))*( 1 + (tgt.rz/(tgt.rg+prm.rho-tgt.rz))*exp(-(tgt.rg+prm.rho)*tgt.Tk) - ((tgt.rg+prm.rho)/(tgt.rg+prm.rho-tgt.rz))*exp(-tgt.rz*tgt.Tk) );
Kbar = (tgt.PP/(1 - tgt.ii + tgt.ii*tgt.PPi_PPh))*((tgt.rkbar-tgt.rg)*tgt.l/((tgt.rg+prm.rho)*(1-exp(-tgt.rkbar*tgt.Tk))))*( 1 + (tgt.rkbar/(tgt.rg+prm.rho-tgt.rkbar))*exp(-(tgt.rg+prm.rho)*tgt.Tk) - ((tgt.rg+prm.rho)/(tgt.rg+prm.rho-tgt.rkbar))*exp(-tgt.rkbar*tgt.Tk) );
Z_Kbar = Z/Kbar;

% Package information
tgtk = struct('kappa',kappa,'Z_Kbar',Z_Kbar);

% Find standard deviation of distribution of credit costs
[prm.sig,~,exitflag] = fzero(@calibcritcred,1,[],tgtk);
if (exitflag < 1)
    disp('Calibration failed: Cannot find credit-cost standard deviation');
    return;
end

% Unconditional mean of distribution of credit costs
prm.mu = log(Z) - prm.sig*norminv(kappa);

%% Search and entry costs

% Entry costs
prm.E = tgt.BBl + kappa*(Z - Kbar);

% Flow search costs
prm.Fh = (tgt.PP/(365*(1 - tgt.ii + tgt.ii*tgt.PPi_PPh)*tgt.PPh_Y))*(tgt.LLambdah/tgt.Tbh)*tgt.Fh_vo_Y;
prm.Fi = tgt.Fi_Fh*prm.Fh;
prm.Fl = tgt.LLambdal_LLambdah*(tgt.Tbh/Tbl)*tgt.Fl_vl_Fh_vo*prm.Fh;

% Extras
PPh_PP = 1/(1 - tgt.ii + tgt.ii*tgt.PPi_PPh);
PPi_PP = tgt.PPi_PPh/(1 - tgt.ii + tgt.ii*tgt.PPi_PPh);
Fh_PPh = prm.Fh/(PPh_PP*tgt.PP);
Fi_PPi = prm.Fi/(PPi_PP*tgt.PP);

%% Discount rate and bargaining powers

% Package information
tgtr = struct('rhol',prm.rhol,'ps',ps,'thetao',thetao,'PPi_PP',PPi_PP,'Fi_PPi',Fi_PPi,'D_PP',tgt.D_PP,'Co_PP',tgt.Co_PP, ...
    'Ci_PPi',tgt.Ci_PPi,'Tbh',tgt.Tbh,'LLambdah',tgt.LLambdah,'PPi_PPh',tgt.PPi_PPh,'Tso',tgt.Tso,'Tsl',tgt.Tsl, ...
    'PPi_RR',tgt.PPi_RR,'Dl_RR',tgt.Dl_RR,'Tml',tgt.Tml,'A_Cl',tgt.A_Cl,'Cl_RR',tgt.Cl_RR,'taui',tgt.taui);

% Find standard deviation of distribution of credit costs
[prm.r,~,exitflag] = fzero(@calibcritr,0.05,[],tgtr);
if (exitflag < 1)
    disp('Calibration failed: Cannot find discount rate');
    return;
end

% Public expenditure relative to house prices
g_PPh = (tgt.tauh*(1 - tgt.ii) + tgt.taui*tgt.ii*tgt.PPi_PPh)*uo/(tgt.n*tgt.Tso);

% Bargaining powers
oomegah = (1 + tgt.tauh)*(1/tgt.Tbh)*( (1 - tgt.Co_PP/PPh_PP)*prm.r + (tgt.D_PP/PPh_PP) + thetao*ps*(tgt.LLambdah/tgt.Tbh)*( 1 - tgt.PPi_PPh ) )/( ...
    ( Fh_PPh - g_PPh + (prm.r + prm.rho)*tgt.BBl + (prm.r + prm.rho)*(Z/(tgt.PP*PPh_PP)) )*( prm.r + (1/tgt.Tso) ));
oomegal = (1/Tbl)*(tgt.PPi_PPh/tgt.PPi_RR)*( 1 - tgt.Dl_RR - (prm.r + (1/tgt.Tml))*(1 - tgt.A_Cl)*tgt.Cl_RR )/( ...
    ( (prm.Fl/prm.Fh)*Fh_PPh - g_PPh + (prm.r + prm.rho)*tgt.BBl )*( prm.r + (1/tgt.Tml) + (1/tgt.Tsl) ));
oomegai = (1 + tgt.taui)*(tgt.LLambdah/tgt.Tbh)*( (1 - (tgt.Co_PP/PPh_PP))*prm.r + (tgt.D_PP/PPh_PP) - (1 - tgt.PPi_PPh)*( ...
    prm.r + thetao*(1-ps)*(1/tgt.Tbh) ) )/( (prm.Fi/prm.Fh)*Fh_PPh*( prm.r + (1/tgt.Tso) ) );
prm.omegah = oomegah/(1+oomegah);
prm.omegal = oomegal/(1+oomegal);
prm.omegai = oomegai/(1+oomegai);
if ((prm.r <= 0) || (prm.omegah <= 0) || (prm.omegah >=1 ) || (prm.omegai <= 0) || (prm.omegai >= 1) || (prm.omegal <= 0) || (prm.omegal >= 1))
    disp('Calibration failed: Discount rate or bargaining power parameters are not well defined');
    return;
end

% Tenants' moving cost
prm.Cw = ( (1/prm.omegal)*(1 - tgt.A_Cl) - 1 )*prm.Cl;
if (prm.Cw < 0)
    disp('Calibration failed: Cannot find tenant moving cost');
    return;
end

%% Meeting functions

% Meeting function elasticities
omegao = (1-ps)*prm.omegah + ps*prm.omegai;
prm.etao = omegao/tgt.omegao_etao;
prm.etal = prm.omegal/tgt.omegal_etal;

% Meeting function productivity paramaters
prm.upsilono = (thetao^prm.etao)*tgt.LLambdah/tgt.Tbh;
prm.upsilonl = (thetal^prm.etal)*tgt.LLambdal_LLambdah*tgt.LLambdah/Tbl;

%% Rental-market parameters

% Rental-market match quality parameters
prm.lambdal = 1 + (((1 - prm.omegal + prm.omegal*thetal)/Tbl)/(prm.r + (1/tgt.Tml))) + prm.omegal*(( tgt.Dl_RR - (prm.Fl/prm.Fi)*Fi_PPi*tgt.PPi_RR ...
    + (prm.r + (1/tgt.Tml))*(1 - tgt.A_Cl)*(tgt.Cl_RR/prm.omegal) - prm.xi*ml*kappa*((Z - Kbar)/tgt.PP)*(tgt.PPi_RR/PPi_PP)  )*( prm.r ...
    + (1/tgt.Tml) + (1/tgt.Tsl) ))/(( 1 - tgt.Dl_RR - (prm.r + (1/tgt.Tml))*(1-tgt.A_Cl)*tgt.Cl_RR )*( prm.r + (1/tgt.Tml)));
prm.zetal = ( prm.Dl - prm.Fl + (prm.r + (1/tgt.Tml))*(prm.Cl + prm.Cw) - prm.xi*ml*kappa*(Z - Kbar) + (tgt.PP*PPi_PP/tgt.PPi_RR)*(1/Tbl)*( ...
    (1 - prm.omegal + prm.omegal*thetal)/prm.omegal)*(( 1 - tgt.Dl_RR - (prm.r + (1/tgt.Tml))*(1 - tgt.A_Cl)*tgt.Cl_RR )/( ... 
    prm.r + (1/tgt.Tml) + (1/tgt.Tsl) )) )/( (tgt.LLambdal_LLambdah*tgt.LLambdah)^(1/prm.lambdal) );
if ((prm.lambdal <= 1) || (prm.zetal <= 0))
    disp('Calibration failed: Rental-market match quality distribution not well defined');
    return;
end

%% Moving decisions and the size of idiosyncratic shocks

% Arrival rate of idiosyncratic shocks for homeowners
prm.alphah = (1 + tgt.varkappa)*mh + prm.rho*tgt.varkappa;

% Home-buyer match quality parameters
omegash = prm.omegah/(1 + tgt.tauh*(1 - prm.omegah));
xh = (tgt.PP*PPh_PP)*( (1/tgt.Tbh)*((1 - omegash + (1-ps)*omegash*thetao)/omegash)*(( (1 - (tgt.Co_PP/PPh_PP))*prm.r + (tgt.D_PP/PPh_PP) ...
    + thetao*ps*(tgt.LLambdah/tgt.Tbh)*(1 - tgt.PPi_PPh) )/( prm.r + (1/tgt.Tso))) + ps*thetao*(tgt.LLambdah/tgt.Tbh)*(( ... 
    (1 - (tgt.Co_PP/PPh_PP))*prm.r + (tgt.D_PP/PPh_PP) - (prm.r + thetao*(1-ps)*(1/tgt.Tbh))*(1 - tgt.PPi_PPh) )/( prm.r + (1/tgt.Tso))) ) - prm.Fh;
if (xh <= 0)
    disp('Calibration failed: Ownership-market moving threshold is not well defined');
    return;
end
yh = xh + (prm.r + prm.rho + prm.alphah)*( prm.Ch + prm.Co + tgt.tauh*( prm.Co - (prm.D/prm.r) + (thetao/prm.r)*(tgt.PP*PPh_PP/tgt.Tbh)*( ...
    (1-ps)*(( (1 - (tgt.Co_PP/PPh_PP))*prm.r + (tgt.D_PP/PPh_PP) + thetao*ps*(tgt.LLambdah/tgt.Tbh)*(1 - tgt.PPi_PPh) )/( prm.r + (1/tgt.Tso)) ) ...
    + ps*tgt.LLambdah*(( (1 - (tgt.Co_PP/PPh_PP))*prm.r + (tgt.D_PP/PPh_PP) - (prm.r + thetao*(1-ps)*(1/tgt.Tbh))*(1 - tgt.PPi_PPh) )/( ... 
    prm.r + (1/tgt.Tso))) ) ));
xh_yh = xh/yh;
yh_PPh = (yh/tgt.PP)/PPh_PP;

% Package information
tgto = struct('r',prm.r,'rho',prm.rho,'alphah',prm.alphah,'omegash',omegash,'yh_PPh',yh_PPh,'xh_yh',xh_yh,'PPh_PP',PPh_PP,'thetao',thetao,'ps',ps, ...
    'Tso',tgt.Tso,'varkappa',tgt.varkappa,'tauh',tgt.tauh,'Co_PP',tgt.Co_PP,'D_PP',tgt.D_PP,'LLambdah',tgt.LLambdah,'Tbh',tgt.Tbh,'PPi_PPh',tgt.PPi_PPh);

% Find lambdah
[prm.lambdah,~,exitflag] = fzero(@calibcritown,30,[],tgto);
if (exitflag < 1)
    disp('Calibration failed: Cannot find home-buyer match quality distribution');
    return;
end

% Size of idiosyncratic shocks faced by homeowners
prm.deltah = (((prm.rho + prm.alphah)*tgt.varkappa)/(prm.alphah*( tgt.varkappa + ((1/xh_yh)^prm.lambdah) )))^(1/prm.lambdah);
if (prm.deltah*yh >= xh)
    disp('Calibration failed: Idiosyncratic shock for homeowners is too small');
    return;
end

% Minimum match quality for new home-buyers
prm.zetah = yh/((tgt.LLambdah)^(1/prm.lambdah));

% Tax rates
prm.tauh = tgt.tauh;
prm.taui = tgt.taui;

% Store initial values for numerical search when solving the model
prm.pt0 = [ps;thetao];

disp('Calibration successful');

end
