
// *********************************************************************************************************************
//* Table A.3 Descriptive statistics for the sample

// * This file contains the code to generate the descriptive statistics of the data.

// It generates a version of :
// that includes panels for the suburbs, the City of Toronto, and the whole sample.

// Global variables used in this file:
// - border: 0 for all data, border = 1 is data excluding the border 3 km, border = 2 is data excluding the border 5 km


// Sample definitions: 
// - sample2 is the sample for the results (Jan 2007 to Feb 2009)
// - sample3 is the main sample for the paper (Jan 2006 to Feb 2012)
// - sample4 is the sample for the robustness checks (Jan 2006 to Feb 2010)
// - sample5 is the sample for the extended analysis (Jan 2006 to Feb 2014)
// - sample6 is the sample for the extended analysis (Jan 2006 to Feb 2018)
// - sample8 is the sample for the extended analysis (Jan 2006 to Feb 2016)


// *********************************************************************************************************************
// * Table A.3 Descriptive statistics 

// Variables:
// - BTO_SaleCount: Number of BTO sales per year
// - BTR_SaleCount: Number of BTR sales per year
// - price_rent_avg_ratio: Price-rent ratio
// - Time-on-market (TOM) 
// - Sale price

// Samples:
// for: 
// - suburb: Suburbs
// - toronto: City of Toronto
// - whole: Whole Sample

// Sample periods:
// - Pre-LTT: 2006:1-2008:1
// - Post-LTT: 2008:1-2010:2
// - Post-LTT: 2010:3-2012:2
// - Whole sample: 2006:1-2018:2

// *********************************************************************************************************************

di "Border on: `border'"

// * Config
if $dominant_log==0{
    cap log close  
    log using "$path_results/05_tableA3_desc_border$border.log", replace
}

local border = $border // border = 0 is all data, border = 1 is data excluding the border 3 km, border = 2 is data excluding the border 5 km

if `border'==1{
    local add = "_border3km"
    local cond_dist = "if abs(distance)<3000"
}
else if `border'==2{
    local add = "_border5km"
    local cond_dist = "if abs(distance)<5000"
}
else{
    local add = ""
}

// *********************************************************************************************************************
// * Open file


local filename = "tableA3_`add'"

cap file close out
file open out using "$path_table/`filename'.tex", write replace
	
file write out 	///
    "\begin{table}[htpb]" _n	///
    "\centering" _n	///
    "\caption{Descriptive statistics}" _n	///
    _tab "\begin{tabular}{lcccc}" _n	///
    _tab "\hline\hline" _n	///
    _tab "& Pre-LTT & \multicolumn{2}{c}{Post-LTT} & Whole Sample \\" _n	///
    _tab " & 2006:1-2008:1 & 2008:2-2010:2  & 2008:1-2012:2 & 2006:1-2018:2 \\ " _n 


// *********************************************************************************************************************
// * Generate samples, config columns and panels

// * col 1 = pre-LTT
// * col 2 = post-LTT - 2008:1-2010:2
// * col 3 = post-LTT - 2010:3-2012:2
// * col 4 = Whole sample - 2006:1-2018

loc columns col1 col2 col3 col4
// loc columns col1

local panels suburb toronto whole 

// *********************************************************************************************************************
// *  Iterating over the panels

foreach s in `panels'{

    di "****************************************************************************************"
    di "Panel: `s'"

    file write out "\hline" _n


    // *********************************************************************************************************************
    // * Market data 
    // *********************************************************************************************************************

    // *********************************************************************************************************************
    // * Load data: Market data

    use "$path_data/Community_18M_distance.dta", clear


    // * Generate variables

    gen BTO_SaleCount_1 = BTO_SaleCount+1

    gen BTO_Sale_Share = BTO_SaleCount/Total_SaleCount
    replace BTO_Sale_Share = 0 if BTO_Sale_Share==.


    gen leasesales_ratio = TotalLease_LeaseCount/Total_SaleCount
    gen BTS_BTO_price_ratio = BTS_AvgSP/BTO_AvgSP
    // gen price_rent_avg_ratio = Total_AvgSP/TotalLease_AvgLeasePrice
    gen price_rent = Total_PRRatio_Avg
    replace price_rent = 0 if price_rent==.

    // * new added Sales to listing ratio
    gen sales_listings = Total_SaleCount/Tot_NewListings

    if `border'>0{
        qui keep `cond_dist'
    }


    // *********************************************************************************************************************
    // * Generate columns and panels

    gen col1 = 1*(post==0 & sample4==1)
    gen col2 = 1*(post==1 & sample4==1)
    gen col3 = 1*(post==1 & sample3==1)
    gen col4 = 1*(sample6==1)

    gen suburb = 1*(toronto==0)
    gen whole = 1

    // *********************************************************************************************************************
    // * variables labels 

    lab var  BTO_SaleCount "\# BTO sales per year"
    lab var  BTR_SaleCount "\# BTR sales per year"

    lab var price_rent "Price-rent ratio"
    lab var sales_listings "Sales-listings ratio"

    lab var  suburb "Suburbs"
    lab var  toronto "City of Toronto"
    lab var  whole "Whole Sample"

    // *********************************************************************************************************************

    // * Mrt Segment Vars panels

        file write out " \multicolumn{5}{c}{\textit{`: var label `s''}} \\" _n

        // preserve

        qui keep if `s'==1

        // * BTR, BTO per sale 

        local col = "col1"

        foreach var in BTO_SaleCount BTR_SaleCount{

            di "******** var: `var'"

            foreach col in `columns'{
                
                qui unique year if `col'==1
                // return list
                local nyears = r(unique)-1

                egen sum1 = total(`var') if `col'==1
                qui sum `v', d
                local tot`col' = "`r(mean)'"

                // by year
                local tot`col' ="`: di %12.0fc `tot`col''/`nyears''"

                qui drop sum1
            }
            file write out "`: var label `var'' & `totcol1' & `totcol2' & `totcol3' & `totcol4' \\" _n
            // file write tbl `filename'  "\\\hline" _newline;
        }

        // * Price rent ratio mean and median

        local format_price_rent = "4.1"
        local format_sales_listings = "4.2"

        foreach var in price_rent sales_listings{

            di "******** var: `var'"
            di "format: `format_`var''f "

            foreach col in `columns'{
                
                qui sum `var' if `col'==1  & `var'!=0 ///
                    , d
                local mean_`col' = "`: di %`format_`var''f r(mean)'"
                local median_`col' = "`: di %`format_`var''f r(p50)'"
                di ": di %`format_`var''f r(p50)"
                di "mean: `mean_`col''"
                di "median: `median_`col''"

            }
            local label_`var'_median = "`: var label `var'' (median)"
            local label_`var'_mean = "`: var label `var'' (mean)"

            local row_mean_`var' = "`label_`var'_mean' & `mean_col1' & `mean_col2' & `mean_col3' & `mean_col4' \\"
            local row_median_`var' = "`label_`var'_median' & `median_col1' & `median_col2' & `median_col3' & `median_col4' \\"

        }

        // restore
    // *********************************************************************************************************************
    // * Transaction level TOM and Sale price
    // *********************************************************************************************************************

    // *********************************************************************************************************************
    // * Load data: Transaction level TOM and Sale price

    use "$path_data/GTA_CLEAN_2000_2018_distance.dta", clear

    if `border'>0{
        qui keep `cond_dist'
    }

    // *********************************************************************************************************************
    // * Generate columns and panels

    gen col1 = 1*(post==0 & sample4==1)
    gen col2 = 1*(post==1 & sample4==1)
    gen col3 = 1*(post==1 & sample3==1)
    gen col4 = 1*(sample6==1)

    gen suburb = 1*(toronto==0)
    gen whole = 1

    // *********************************************************************************************************************
    // * variables labels

    lab var  dom "Time on the market"
    lab var  sold_ "Sale price"

    // *********************************************************************************************************************
        // preserve

        qui keep if `s'==1

        // * mean and median
        foreach var in dom sold_{

            di "******** var: `var'"

            if "`var'"=="dom"{

                local fmt = "%4.1f"
                local label_`var'_median = "`: var label `var'' (median, days)"
                local label_`var'_mean = "`: var label `var'' (mean, days)"
            }
            else{
                local fmt = "%12.0fc"
                local label_`var'_median = "`: var label `var'' (median)"
                local label_`var'_mean = "`: var label `var'' (mean)"
            }
            foreach col in `columns'{

                qui sum `var' if `col'==1 & `var'!=0 ///
                    , d
                local mean_`col' = "`: di `fmt' r(mean)'"
                // local sd_`col' = "`: di %4.1f r(sd)'"
                local median_`col' = "`: di `fmt' r(p50)'"

                di "mean: `mean_`col''"
                di "median: `median_`col''"

            }
        file write out "`label_`var'_mean' & `mean_col1' & `mean_col2' & `mean_col3' & `mean_col4' \\" _n
        file write out "`label_`var'_median' & `median_col1' & `median_col2' & `median_col3' & `median_col4' \\" _n
            
        }

        // restore
    // *********************************************************************************************************************
    // * Price rent at the end

        // local var = "price_rent"
        foreach var in price_rent sales_listings{

            file write out "`row_mean_`var''" _n

            file write out "`row_median_`var''" _n
        }

} // * end panels

// *********************************************************************************************************************
//* Close table

local notes "\textit{Source:} Multiple Listing Service (MLS) residential records (2006-2018)." 

file write out "\hline \hline" _n
file write out "\end{tabular}" _n
//    \begin{minipage}{\textwidth}
//       \footnotesize{\textit{Notes:} ... //   

file write out  "\begin{minipage}{\textwidth} \footnotesize{`notes' \end{minipage}" _n

file write out "\end{table}" _n

file close out

//  *********************************************************************************************************************


if $dominant_log==0{
    log close
}